<?php

namespace Drupal\commerce_paystar\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Exception\InvalidResponseException;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Url;

/**
 * Provides the Off-site Redirect payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "paystar_offsite_redirect",
 *   label = "PayStar (Off-site redirect)",
 *   display_label = "PayStar",
 *   forms = {
 *     "offsite-payment" =
 *   "Drupal\commerce_paystar\PluginForm\OffsiteRedirect\PaymentOffsiteForm",
 *   },
 *   payment_method_types = {"credit_card"}
 * )
 */
class OffsiteRedirect extends OffsitePaymentGatewayBase
{
	public function defaultConfiguration()
	{
		return array('terminal' => '') + parent::defaultConfiguration();
	}

	public function buildConfigurationForm(array $form, FormStateInterface $form_state)
	{
		$form = parent::buildConfigurationForm($form, $form_state);
		$form['terminal'] = array(
				'#type' => 'textfield',
				'#title' => t('Terminal ID'),
				'#default_value' => $this->configuration['terminal'],
				'#description' => t("The Terminal ID which is provided by PayStar"),
				'#required' => TRUE,
			);
		return $form;
	}

	public function submitConfigurationForm(array &$form, FormStateInterface $form_state)
	{
		parent::submitConfigurationForm($form, $form_state);
		if (!$form_state->getErrors())
		{
			$values = $form_state->getValue($form['#parents']);
			$this->configuration['terminal'] = $values['terminal'];
		}
	}

	public function onReturn(OrderInterface $order, Request $request)
	{
		require_once(drupal_get_path('module', 'commerce_paystar') . '/paystar/paystar_payment_helper.class.php');
		$p = new \PayStar_Payment_Helper($this->configuration['terminal']);
		$r = $p->paymentVerify($x=array_merge($_POST, array('amount' => intval($order->getTotalPrice()->getNumber()))));
		if ($r)
		{
			$query = \Drupal::entityQuery('commerce_payment')->condition('remote_state', $p->txn_id);
			$payments = $query->execute();
			if (count($payments) > 0)
			{
				\Drupal::logger('commerce_paystar')->error('commerce_paystar: Double spending occured on order <a href="@url">%order</a> from ip @ip', array(
						'@url'   => Url::fromUri('base:/admin/commerce/orders/' . $order->id())->toString(),
						'%order' => $order->id(),
						'@ip'    => $order->getIpAddress(),
					));
				$this->messenger()->addMessage('Double spending occured.');
				$checkout_flow = $order->checkout_flow->entity;
				$checkout_flow_plugin = $checkout_flow->getPlugin();
				$redirect_step = $checkout_flow_plugin->getPreviousStepId('payment');
				$checkout_flow_plugin->redirectToStep($redirect_step);
			}
			else
			{
				$payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
				$payment = $payment_storage->create(array(
						'state'           => 'completed',
						'amount'          => $order->getTotalPrice(),
						'payment_gateway' => $this->entityId,
						'order_id'        => $order->id(),
						'test'            => false,
						'remote_id'       => $p->txn_id,
						'authorized'      => $this->time->getRequestTime(),
					));
				$payment->save();
				$this->messenger()->addMessage($this->t('Payment was processed.') . ' ' . $this->t('Transaction ID : ') . $p->txn_id);
			}
		}
		else
		{
			$message = $this->t("commerce_paystar: Payment failed : %error", ['%error' => $p->error]);
			$redirect = Url::fromRoute('commerce_payment.checkout.cancel', ['commerce_order' => $order->id(), 'step' => 'payment'], ['absolute' => TRUE]);
			$this->messenger()->addError($message);
			$this->messenger()->addMessage($message);
			header('location: ' . $redirect->toString());exit;
			throw new PaymentGatewayException($message);
		}
	}
}
