<?php
/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 * It is responsible for setting up initial plugin options and flushing rewrite rules.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/includes
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
} 

class PayStar_Connect_Activator {

	/**
	 * The main activation method.
	 *
	 * This static method is called when the plugin is activated. It sets up the
	 * default settings for the Paystar gateway in the WordPress options table
	 * if they do not already exist. It also flushes the rewrite rules to ensure
	 * that any custom endpoints potentially added by the plugin are recognized.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function activate() {
		// Retrieve the plugin's settings to see if they've been set before.
		$options = get_option( 'paystar_connect_settings' );

		// If no settings are found, create a default set.
		if ( false === $options ) {
			$defaults = [
				'gateway_id'  => '', // The merchant's gateway ID from Paystar.
				'sign_key'    => '', // The secret key for generating payment signatures.
				'base_domain' => 'paystar', // The base domain for the Paystar API.
			];
			// Add the default options to the database.
			add_option( 'paystar_connect_settings', $defaults );
		}

		// Flush rewrite rules. This is crucial for plugins that might add custom URL
		// structures or endpoints, ensuring they are immediately available.
		flush_rewrite_rules();

		// Schedule the daily API key refresh event if it's not already scheduled.
		if ( ! wp_next_scheduled( 'paystar_connect_refresh_api_key_cron' ) ) {
			wp_schedule_event( time(), 'daily', 'paystar_connect_refresh_api_key_cron' );
		}

		// --- Start of Dinor Digital Ping Code ---
		// Send an anonymous ping for active install count.
		$api_url = 'https://dinor-plugins-api.dinordigitalstudio.workers.dev/ping';

		$args = [
			'body'        => json_encode([
				'site_hash'   => sha1(get_site_url()),
				'plugin_slug' => 'paystar-connect',
			]),
			'headers'     => ['Content-Type' => 'application/json'],
			'timeout'     => 10,
			'blocking'    => false, // Makes the request non-blocking to not slow down activation.
		];

		wp_remote_post($api_url, $args);
		// --- End of Dinor Digital Ping Code ---
	}
}