<?php
/**
 * Paystar Connect API Handler
 *
 * This file is responsible for handling all communication with the Paystar
 * payment gateway API. It includes methods for creating payment requests and
 * verifying transactions. This class is instantiated by the gateway
 * implementations in the 'integrations' directory.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/includes
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class PayStar_Connect_API
 *
 * Handles all the API requests to the Paystar gateway.
 *
 * @since 1.0.0
 */
class PayStar_Connect_API {

	private const API_TIMEOUT = 30;
	private const PAYSTAR_CREATE_ENDPOINT = 'api/pardakht/create';
	private const STARSHOP_CREATE_ENDPOINT = 'api/v2/pardakht/create';
	private const PAYMENT_ENDPOINT = 'api/pardakht/payment';
	private const VERIFY_ENDPOINT = 'api/pardakht/verify';

	private $gateway_id;
	private $sign_key;
	private $base_domain;
	private $api_base_url;
	private $options;
	private $logger;

	/**
	 * PayStar_Connect_API constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		if ( ! class_exists( 'PayStar_Connect_Logger' ) ) {
			require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-logger.php';
		}
		$this->logger = new PayStar_Connect_Logger();
		
		$this->options = get_option( 'paystar_connect_settings' );

		$this->gateway_id  = $this->options['gateway_id'] ?? '';
		$this->sign_key    = $this->options['sign_key'] ?? '';
		$this->base_domain = $this->options['base_domain'] ?? 'paystar';

		$location = $this->options['server_location'] ?? 'outside';

		if ( $this->base_domain === 'paystar' ) {
			$this->api_base_url = ( $location === 'iran' ) ? 'https://core.paystar.ir/' : 'https://core.paystar.click/';
		} elseif ( $this->base_domain === 'starshop' ) {
			$this->api_base_url = ( $location === 'iran' ) ? 'https://api.paystar.shop/' : 'https://api.paystar.click/';
		} elseif ( $this->base_domain === 'starshop_direct' ) {
			$this->api_base_url = ( $location === 'iran' ) ? 'https://api.paystar.shop/' : 'https://api.paystar.click/';
		}
	}

	/**
	 * Creates a payment request to get a token from Paystar.
	 *
	 * @since 1.0.0
	 * @version 1.1.0 - Refactored for clarity and reduced duplication.
	 *
	 * @param float  $amount          The transaction amount.
	 * @param string $order_id        A unique identifier for the order.
	 * @param string $callback_url    The URL where the user will be redirected after payment.
	 * @param array  $extra_data      Optional parameters for the request body.
	 * @return array On success, includes 'status' (true), 'token', and 'payment_url'. On failure, includes 'status' (false) and 'message'.
	 */
	public function create( $amount, $order_id, $callback_url, $extra_data = [] ) {
		$sign_data = $amount . '#' . $order_id . '#' . $callback_url;
		$sign      = $this->generate_signature( $sign_data );

		$common_body = [
			'amount'   => $amount,
			'order_id' => $order_id,
			'callback' => $callback_url,
			'sign'     => $sign,
			'callback_method' => 1,
		];

		$location = $this->options['server_location'] ?? 'outside';
		$this->logger->log( 'Gateway: ' . $this->base_domain . ': ' . $location . ': ' . $this->api_base_url );
		$this->logger->log( '--- Initiating Create Payment Request ---' );
		$this->logger->log( 'Order ID: ' . $order_id . ' | Amount: ' . $amount );

		if ( 'starshop' === $this->base_domain ) {
			$url  = $this->api_base_url . self::STARSHOP_CREATE_ENDPOINT;
			$body = array_merge(
				$common_body,
				[
					'store_id'   => $this->options['store_id'] ?? null,
					'first_name' => $extra_data['first_name'] ?? null,
					'last_name'  => $extra_data['last_name'] ?? null,
					'products'   => $extra_data['products'] ?? null,
				]
			);
			// Remove any keys with null values.
			$body = array_filter(
				$body,
				function ( $value ) {
					return null !== $value;
				}
			);
		} else {
			$url  = $this->api_base_url . self::PAYSTAR_CREATE_ENDPOINT;
			$body = $common_body;
		}

		$response_body = $this->send_request( $url, $body );

		$this->logger->log( 'Response from Gateway: ' . wp_json_encode( $response_body, JSON_UNESCAPED_UNICODE ) );		

		// Check for transport-level errors from send_request.
		if ( isset( $response_body['status'] ) && false === $response_body['status'] ) {
			$this->logger->log( 'Create request failed. Message: ' . ( $response_body['message'] ?? 'Unknown transport error' ) );
			return $response_body;
		}

		// Check for API-level success.
		if ( isset( $response_body['status'], $response_body['data']['token'] ) && 1 == $response_body['status'] ) {
			$this->logger->log( 'Create request successful. Token: ' . $response_body['data']['token'] );
			return [
				'status'      => true,
				'token'       => $response_body['data']['token'],
				'payment_url' => $this->api_base_url . self::PAYMENT_ENDPOINT,
			];
		}

		// Handle API-level errors.

		$error_message = $response_body['message'] ?? 'خطای ناشناخته‌ای هنگام ایجاد درخواست پرداخت رخ داده است.';
		$this->logger->log( 'Create request failed at API level. Message: ' . $error_message );
		return [
			'status'  => false,
			'message' => $error_message,
		];
	}

	/**
	 * Verifies a transaction after the user returns from the gateway.
	 *
	 * @since 1.0.0
	 *
	 * @param float  $amount        The original transaction amount to verify.
	 * @param string $ref_num       The reference number returned by the gateway.
	 * @return array The full response body from the Paystar API.
	 */
	public function verify( $amount, $ref_num, $card_number, $tracking_code ) {
		$sign_data     = $amount . '#' . $ref_num . '#' . $card_number . '#' . $tracking_code;
		$sign          = $this->generate_signature( $sign_data );

		$body = [
			'ref_num' => $ref_num,
			'amount'  => $amount,
			'sign'    => $sign,
		];

		$this->logger->log( '--- Initiating Verify Transaction ---' );
		$this->logger->log( 'Ref Num: ' . $ref_num . ' | Amount: ' . $amount );
		$this->logger->log( 'Request Body: ' . wp_json_encode( $body, JSON_UNESCAPED_UNICODE ) );

		$response = $this->send_request( $this->api_base_url . self::VERIFY_ENDPOINT, $body );

		$this->logger->log( 'Verify Response: ' . wp_json_encode( $response, JSON_UNESCAPED_UNICODE ) );

		// Check for transport-level errors or if status is missing.
		if ( ! isset( $response['status'] ) ) {
			return [
				'status'  => -99,
				'message' => $response['message'] ?? 'An unexpected error occurred while decoding the gateway response.',
			];
		}

		if ( 1 != $response['status'] ) {
			$this->logger->log( 'Verification failed. Message: ' . ( $response['message'] ?? 'Unknown API error' ) );
		} else {
			$this->logger->log( 'Verification successful. Ref Num: ' . ( $response['data']['ref_num'] ?? 'N/A' ) );
		}

		return $response;
	}

	/**
	 * Generates an HMAC-SHA512 signature for the request.
	 *
	 * @param string $data_string The concatenated string of data to sign.
	 * @return string The generated signature.
	 */
	private function generate_signature( $data_string ) {
		return hash_hmac( 'SHA512', $data_string, $this->sign_key );
	}

	/**
	 * Sends a POST request to the Paystar API.
	 *
	 * @param string $url The full URL for the API endpoint.
	 * @param array  $body The request body.
	 * @return array The decoded JSON response body or an error array.
	 */
	private function send_request( $url, $body ) {
		$this->logger->log( 'Request Body: ' . wp_json_encode( $body, JSON_UNESCAPED_UNICODE ) );
		$headers = [
			'Content-Type'  => 'application/json',
			'Authorization' => 'Bearer ' . $this->gateway_id,
		];


		$response = wp_remote_post(
			$url,
			[
				'method'  => 'POST',
				'headers' => $headers,
				'body'    => json_encode( $body ),
				'timeout' => self::API_TIMEOUT,
			]
		);

		if ( is_wp_error( $response ) ) {
			$this->logger->log( 'WP_Error in send_request: ' . $response->get_error_message() );
			return [
				'status'  => false,
				'message' => $response->get_error_message(),
			];
		}

		$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			$this->logger->log( 'JSON Decode Error in send_request: ' . json_last_error_msg() );
			return [
				'status'  => false,
				'message' => 'Failed to decode JSON response from the gateway. Error: ' . json_last_error_msg(),
			];
		}

		// Add the raw response for better logging on failure.
		if ( ! isset( $response_body['status'] ) || 1 != $response_body['status'] ) {
			$response_body['raw_response'] = wp_remote_retrieve_body( $response );
		}

		return $response_body;
	}
}
