<?php
/**
 * Paystar Connect Settings Page
 *
 * This file is responsible for creating and managing the main settings page for the
 * Paystar Connect plugin within the WordPress admin area. It handles the registration
 * of settings, sections, and fields required to configure the Paystar payment gateway.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/includes
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
} 

/**
 * Handles the creation of the plugin's settings page in the WordPress admin area.
 *
 * This class registers the admin menu, initializes settings sections and fields,
 * and renders the HTML for the settings page.
 *
 * @since 1.0.0
 */
class PayStar_Connect_Settings {

	/**
	 * Holds the plugin's settings options.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    array $options The stored settings from the database.
	 */
	private $options;

	/**
	 * Constructor for the settings class.
	 *
	 * Hooks the necessary actions into WordPress to create the admin menu and
	 * initialize the settings fields.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'admin_init', [ $this, 'page_init' ] );
		add_action( 'admin_post_paystar_download_log', [ $this, 'handle_download_log' ] );
		add_action( 'admin_post_paystar_clear_log', [ $this, 'handle_clear_log' ] );
		add_action( 'admin_notices', [ $this, 'show_clear_log_notice' ] );
	}

	/**
	 * Renders the main settings page content.
	 *
	 * This method retrieves the stored options and generates the HTML for the
	 * settings form, including the custom card-based layout.
	 *
	 * @since 1.0.0
	 */
	public function create_admin_page() {
		// Retrieve the saved options to populate the form fields.
		$this->options = get_option( 'paystar_connect_settings' );
		?>
		<div class="wrap paystar-settings-wrap">
			<h1><?php esc_html_e( 'تنظیمات درگاه پرداخت پی‌استار', 'paystar-connect' ); ?></h1>
			<p><?php esc_html_e( 'پیکربندی درگاه پرداخت یکپارچه پی‌استار برای وردپرس', 'paystar-connect' ); ?></p>
			
			<form method="post" action="options.php">
				<?php
					// This function outputs the hidden fields for the settings group.
					settings_fields( 'paystar_connect_option_group' );
				?>
				<div id="paystar-api-card" class="paystar-settings-card">
					<h2 class="paystar-card-header"><?php esc_html_e( 'اطلاعات اتصال به درگاه', 'paystar-connect' ); ?></h2>
					<div class="paystar-card-body">
						<?php
							// Renders the settings fields for the API section.
							do_settings_sections( 'paystar-connect-api-section' );
						?>
					</div>
				</div>

				<div id="paystar-config-card" class="paystar-settings-card">
					<h2 class="paystar-card-header"><?php esc_html_e( 'پیکربندی درگاه', 'paystar-connect' ); ?></h2>
					<div class="paystar-card-body">
						<?php
							// Renders the settings fields for the Configuration section.
							do_settings_sections( 'paystar-connect-config-section' );
						?>
					</div>
				</div>

				<div id="paystar-starshop-card" class="paystar-settings-card">
					<h2 class="paystar-card-header"><?php esc_html_e( 'تنظیمات فروشگاه‌ساز استارشاپ', 'paystar-connect' ); ?></h2>
					<div class="paystar-card-body">
						<?php
						// Renders the settings fields for the Sync section.
						do_settings_sections( 'paystar-connect-sync-section' );
						?>
					</div>
				</div>

				<div id="paystar-supported-plugins-card" class="paystar-settings-card">
					<h2 class="paystar-card-header"><?php esc_html_e( 'افزونه‌های پشتیبانی شده', 'paystar-connect' ); ?></h2>
					<div class="paystar-card-body">
						<?php
							// Renders the informational fields for the Status section.
							do_settings_sections( 'paystar-connect-status-section' );
						?>
					</div>
				</div>
				
				<?php
					// Outputs the standard submit button.
					submit_button();
				?>
				<div id="paystar-debugging-card" class="paystar-settings-card">
					<h2 class="paystar-card-header"><?php esc_html_e( 'عیب‌یابی', 'paystar-connect' ); ?></h2>
					<div class="paystar-card-body">
						<?php do_settings_sections( 'paystar-connect-debug-section' ); ?>
					</div>
				</div>
			</form>
		</div>
		<?php
	}

	/**
	 * Registers all settings, sections, and fields with the WordPress Settings API.
	 *
	 * This method is hooked into 'admin_init' and sets up the structure of the
	 * settings page.
	 *
	 * @since 1.0.0
	 */
	public function page_init() {
		// Register the main setting group.
		register_setting(
			'paystar_connect_option_group', // Option group
			'paystar_connect_settings',     // Option name
			[ $this, 'sanitize' ]          // Sanitization callback
		);

		// Section 1: API Credentials
		add_settings_section(
			'paystar-connect-api-section', // ID
			'',                            // Title (not displayed)
			null,                          // Callback
			'paystar-connect-api-section'  // Page
		);
		add_settings_field( 'gateway_id', 'شناسه درگاه', [ $this, 'gateway_id_callback' ], 'paystar-connect-api-section', 'paystar-connect-api-section' );
		add_settings_field( 'sign_key', 'کلید رمزنگاری', [ $this, 'sign_key_callback' ], 'paystar-connect-api-section', 'paystar-connect-api-section' );

		// Section 2: Gateway Configuration
		add_settings_section( 'paystar-connect-config-section', '', null, 'paystar-connect-config-section' );
		add_settings_field( 'base_domain', 'سرویس‌دهنده', [ $this, 'base_domain_callback' ], 'paystar-connect-config-section', 'paystar-connect-config-section' );
		add_settings_field( 'server_location', 'محل سرور شما', [ $this, 'server_location_callback' ], 'paystar-connect-config-section', 'paystar-connect-config-section' );

		// Section 3: Plugin Status (Informational)
		add_settings_section( 'paystar-connect-status-section', '', null, 'paystar-connect-status-section' );
		add_settings_field( 'supported_in_version', 'در این نسخه', [ $this, 'supported_in_version_callback' ], 'paystar-connect-status-section', 'paystar-connect-status-section' );
		add_settings_field( 'active_on_site', 'فعال در سایت شما', [ $this, 'active_on_site_callback' ], 'paystar-connect-status-section', 'paystar-connect-status-section' );

		// Section 4: Global Sync
		add_settings_section(
			'paystar-connect-sync-section', // ID
			'',                                 // Title
			null,                               // Callback
			'paystar-connect-sync-section'      // Page
		);
		add_settings_field( 'store_id', 'شناسه فروشگاه', [ $this, 'store_id_callback' ], 'paystar-connect-sync-section', 'paystar-connect-sync-section' );
		add_settings_field( 'store_api_key', 'کلید API فروشگاه', [ $this, 'store_api_key_callback' ], 'paystar-connect-sync-section', 'paystar-connect-sync-section' );
		add_settings_field( 'store_refresh_token', 'کلید بازسازی', [ $this, 'store_refresh_token_callback' ], 'paystar-connect-sync-section', 'paystar-connect-sync-section' );
		add_settings_field(
			'sync_controls',
			__( 'همگام‌سازی سراسری', 'paystar-connect' ),
			[ $this, 'sync_controls_callback' ],
			'paystar-connect-sync-section',
			'paystar-connect-sync-section'
		);

		// Section 5: Debugging
		add_settings_section( 'paystar-connect-debug-section', '', null, 'paystar-connect-debug-section' );
		add_settings_field( 'debug_mode', 'حالت عیب‌یابی', [ $this, 'debug_mode_callback' ], 'paystar-connect-debug-section', 'paystar-connect-debug-section' );
		add_settings_field( 'log_management', 'مدیریت فایل لاگ', [ $this, 'log_management_callback' ], 'paystar-connect-debug-section', 'paystar-connect-debug-section' );
	}

	/**
	 * Sanitizes each setting field before saving to the database.
	 *
	 * Ensures that the input values are clean and safe.
	 *
	 * @since  1.0.0
	 * @param  array $input Contains all settings fields as array keys.
	 * @return array The sanitized input.
	 */
	public function sanitize( $input ) {
		$new_input = [];
		if ( isset( $input['gateway_id'] ) ) {
			$new_input['gateway_id'] = sanitize_text_field( $input['gateway_id'] );
		}
		if ( isset( $input['sign_key'] ) ) {
			$new_input['sign_key'] = sanitize_textarea_field( $input['sign_key'] );
		}
		if ( isset( $input['base_domain'] ) ) {
			$new_input['base_domain'] = sanitize_text_field( $input['base_domain'] );
		}
		if ( isset( $input['server_location'] ) ) {
			$new_input['server_location'] = sanitize_text_field( $input['server_location'] );
		}
		if ( isset( $input['store_id'] ) ) {
			$new_input['store_id'] = sanitize_text_field( $input['store_id'] );
		}
		if ( isset( $input['store_api_key'] ) ) {
			$new_input['store_api_key'] = sanitize_text_field( $input['store_api_key'] );
		}
		if ( isset( $input['store_refresh_token'] ) ) {
			$new_input['store_refresh_token'] = sanitize_textarea_field( $input['store_refresh_token'] );
		}
		if ( isset( $input['debug_mode'] ) ) {
			$new_input['debug_mode'] = 'yes' === $input['debug_mode'] ? 'yes' : 'no';
		} else {
			$new_input['debug_mode'] = 'no';
		}

		return $new_input;
	}

	// --- Field Callback Functions --- //

	/**
	 * Renders the input field for the Gateway ID.
	 *
	 * @since 1.0.0
	 */
	public function gateway_id_callback() {
		printf(
			'<input type="text" id="gateway_id" name="paystar_connect_settings[gateway_id]" value="%s" class="regular-text" />',
			isset( $this->options['gateway_id'] ) ? esc_attr( $this->options['gateway_id'] ) : ''
		);
		echo '<p class="description">' . esc_html__( 'این شناسه را از پنل کاربری خود در وب‌سایت پی‌استار دریافت کنید.', 'paystar-connect' ) . '</p>';
	}

	/**
	 * Renders the textarea for the Sign Key.
	 *
	 * @since 1.0.0
	 */
	public function sign_key_callback() {
		printf(
			'<textarea id="sign_key" name="paystar_connect_settings[sign_key]" class="regular-text" rows="5">%s</textarea>',
			isset( $this->options['sign_key'] ) ? esc_textarea( $this->options['sign_key'] ) : ''
		);
		echo '<p class="description">' . esc_html__( 'کلید رمزنگاری (Sign Key) امضای دیجیتال تراکنش‌ها را از قسمت تنظیمات درگاه در پی‌استار دریافت کنید.', 'paystar-connect' ) . '</p>';
	}
	
	/**
	 * Renders the select dropdown for the Base Domain (service provider).
	 *
	 * @since 1.0.0
	 */
	public function base_domain_callback() {
		$current = isset( $this->options['base_domain'] ) ? $this->options['base_domain'] : 'paystar';
		?>
		<select name="paystar_connect_settings[base_domain]">
			<option value="paystar" <?php selected( $current, 'paystar' ); ?>><?php esc_html_e( 'پی‌استار', 'paystar-connect' ); ?></option>
			<option value="starshop_direct" <?php selected( $current, 'starshop_direct' ); ?>><?php esc_html_e( 'استارشاپ', 'paystar-connect' ); ?></option>
			<option value="starshop" <?php selected( $current, 'starshop' ); ?>><?php esc_html_e( 'استارشاپ (اتصال به فروشگاه‌ساز)', 'paystar-connect' ); ?></option>
		</select>
		<p class="description"><?php esc_html_e( 'سرویس‌دهنده‌ای که از آن خدمات پرداخت را دریافت کرده‌اید، انتخاب کنید.', 'paystar-connect' ); ?></p>
		<?php
	}

	/**
	 * Renders the radio buttons for the Server Location.
	 *
	 * This setting helps optimize API calls based on the server's geographical location.
	 *
	 * @since 1.0.0
	 */
	public function server_location_callback() {
		$current = isset( $this->options['server_location'] ) ? $this->options['server_location'] : 'outside';
		?>
		<fieldset>
			<label>
				<input type="radio" name="paystar_connect_settings[server_location]" value="outside" <?php checked( $current, 'outside' ); ?>>
				<span><?php esc_html_e( 'خارج از ایران', 'paystar-connect' ); ?></span>
			</label>
			<br>
			<label>
				<input type="radio" name="paystar_connect_settings[server_location]" value="iran" <?php checked( $current, 'iran' ); ?>>
				<span><?php esc_html_e( 'داخل ایران', 'paystar-connect' ); ?></span>
			</label>
		</fieldset>
		<p class="description"><?php esc_html_e( 'برای بهینه‌سازی سرعت اتصال، نزدیک‌ترین موقعیت به سرور خود را انتخاب کنید.', 'paystar-connect' ); ?></p>
		<?php
	}

	/**
	 * Renders the input field for the Store ID.
	 *
	 * @since 2.1.0
	 */
	public function store_id_callback() {
		printf(
			'<input type="text" id="store_id" name="paystar_connect_settings[store_id]" value="%s" class="regular-text" />',
			isset( $this->options['store_id'] ) ? esc_attr( $this->options['store_id'] ) : ''
		);
		echo '<p class="description">' . wp_kses_post( 'شناسه فروشگاه خود را از <a href="https://paystar.shop/panel" target="_blank">پنل کاربری فروشگاه‌ساز استارشاپ</a> دریافت و در این قسمت وارد کنید.' ) . '</p>';
	}

	/**
		* Renders the input field for the Store API Key.
		*
		* @since 1.0.0
		*/
	public function store_api_key_callback() {
		printf(
			'<input type="text" id="store_api_key" name="paystar_connect_settings[store_api_key]" value="%s" class="regular-text" />',
			isset( $this->options['store_api_key'] ) ? esc_attr( $this->options['store_api_key'] ) : ''
		);
		echo '<p class="description">' . wp_kses_post( 'کلید API فروشگاه خود را از <a href="https://paystar.shop/panel/seller/api-key" target="_blank">صفحه توکن وب‌سرویس</a> در پنل کاربری فروشگاه‌ساز استارشاپ دریافت کنید.' ) . '</p>';
	}

	/**
		* Renders the input field for the Store Refresh Token.
		*
		* @since 1.0.0
		*/
	public function store_refresh_token_callback() {
		printf(
			'<textarea id="store_refresh_token" name="paystar_connect_settings[store_refresh_token]" class="regular-text" rows="3">%s</textarea>',
			isset( $this->options['store_refresh_token'] ) ? esc_textarea( $this->options['store_refresh_token'] ) : ''
		);
		echo '<p class="description">' . wp_kses_post( 'کلید بازسازی را از <a href="https://paystar.shop/panel/seller/api-key" target="_blank">صفحه توکن وب‌سرویس</a> در پنل کاربری فروشگاه‌ساز استارشاپ دریافت کنید.' ) . '</p>';
	}

	/**
	 * Renders a static list of all plugins supported by this version of Paystar Connect.
	 *
	 * This is for informational purposes, showing the user what integrations are available.
	 *
	 * @since 1.0.0
	 */
	public function supported_in_version_callback() {
		echo '<ul>';
		echo '<li><span class="dashicons dashicons-yes" style="color: #444444;"></span> ' . esc_html__( 'ووکامرس', 'paystar-connect' ) . '</li>';
		echo '<li><span class="dashicons dashicons-yes" style="color: #444444;"></span> ' . esc_html__( 'گرویتی‌فرم', 'paystar-connect' ) . '</li>';
		echo '</ul>';
	}

	/**
	 * Renders a dynamic list of supported plugins that are currently active on the site.
	 *
	 * Checks for the existence of key classes from supported plugins (e.g., 'WooCommerce', 'GFCommon')
	 * to determine if they are active.
	 *
	 * @since 1.0.0
	 */
	public function active_on_site_callback() {
		$active_plugins_found = false;
		$output                 = '<ul class="wp-list-with-checks">';

		// Check if WooCommerce is active.
		if ( class_exists( 'WooCommerce' ) ) {
			$url                  = admin_url( 'admin.php?page=wc-settings&tab=checkout' );
			$output              .= '<li><a href="' . esc_url( $url ) . '" target="_blank" style="text-decoration: none">' . esc_html__( 'ووکامرس', 'paystar-connect' ) . '</a></li>';
			$active_plugins_found = true;
		}
		// Check if Gravity Forms is active.
		if ( class_exists( 'GFCommon' ) ) {
			$url                  = admin_url( 'admin.php?page=gf_settings' );
			$output              .= '<li><a href="' . esc_url( $url ) . '" target="_blank" style="text-decoration: none">' . esc_html__( 'گرویتی‌فرم', 'paystar-connect' ) . '</a></li>';
			$active_plugins_found = true;
		}
		
		$output .= '</ul>';

		// If no compatible plugins are found, display a message.
		if ( ! $active_plugins_found ) {
			$output = '<p class="description">' . esc_html__( 'هیچ افزونه سازگاری در سایت شما فعال نیست.', 'paystar-connect' ) . '</p>';
		}

		// The 'echo' is safe here as all content is either static or escaped.
		echo $output; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Renders the controls for the global bulk sync functionality.
	 *
	 * This function generates the HTML for the sync type selector, start/cancel buttons,
	 * and progress bar. It fetches available sync handlers from the Sync Manager.
	 *
	 * @since 2.1.0
	 */
	public function sync_controls_callback() {
		$sync_manager = Paystar_Connect_Sync_Manager::get_instance();
		$handlers     = $sync_manager->get_registered_handlers();
		?>
		<div class="paystar-sync-controls-wrapper">
			<?php if ( empty( $handlers ) ) : ?>
				<p><?php esc_html_e( 'هیچ نوع همگام‌سازی برای نمایش وجود ندارد.', 'paystar-connect' ); ?></p>
			<?php else : ?>
				<p class="description" style="max-width: 700px; margin-bottom: 15px;">
					<?php esc_html_e( 'از این بخش می‌توانید اطلاعات محصولات پشتیبانی‌شده را به صورت دسته‌جمعی با  فروشگاه‌ساز استارشاپ همگام‌سازی کنید. لطفاً محصولات افزونهٔ موردنظر خود را انتخاب و روی دکمه شروع کلیک کنید.', 'paystar-connect' ); ?>
				</p>
				<div class="sync-controls-row">
					<select id="sync-type-selector" class="paystar-admin-select">
						<?php foreach ( $handlers as $key => $handler ) : ?>
							<option value="<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $handler->get_name() ); ?></option>
						<?php endforeach; ?>
					</select>
					<button type="button" id="start-bulk-sync" class="button button-primary">
						<span class="dashicons dashicons-update" style="vertical-align: middle; margin-left: 5px;"></span>
						<?php esc_html_e( 'شروع همگام‌سازی سراسری', 'paystar-connect' ); ?>
					</button>
					<button type="button" id="cancel-bulk-sync" class="button" style="display: none;">
						<?php esc_html_e( 'لغو عملیات همگام‌سازی', 'paystar-connect' ); ?>
					</button>
				</div>
				<div id="sync-progress-bar-container">
					<div id="sync-progress-bar"></div>
					<div id="sync-progress-bar-percentage">0%</div>
				</div>
				<div id="sync-status-text"></div>
				<div id="sync-error-log-container" class="paystar-sync-error-logs" style="display: none; margin-top: 15px;">
					<section class="panel" aria-labelledby="errors-title">
						<div class="panel-header">
							<div id="errors-title" class="panel-title"><?php esc_html_e( 'جزئیات رخدادها', 'paystar-connect' ); ?></div>
						</div>
						<div id="sync-error-log-list" class="list" role="list">
							<!-- Error logs will be dynamically inserted here -->
						</div>
					</section>
				</div>
				<?php wp_nonce_field( 'starshop_bulk_sync_nonce', 'starshop_bulk_sync_nonce_field' ); ?>
			<?php endif; ?>
		</div>
		<?php
	}

	
	/**
	 * Renders the checkbox for the Debug Mode.
	 *
	 * @since 1.3.0
	 */
	public function debug_mode_callback() {
		?>
		<label>
			<input type="checkbox" name="paystar_connect_settings[debug_mode]" value="yes" <?php checked( $this->options['debug_mode'] ?? 'no', 'yes' ); ?> />
			<?php esc_html_e( 'فعالسازی لاگ‌برداری', 'paystar-connect' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'در صورت فعال بودن، تمام رویدادهای مربوط به درگاه پرداخت در یک فایل لاگ ذخیره می‌شود. این فایل ممکن است حاوی اطلاعات حساس باشد.', 'paystar-connect' ); ?>
		</p>
		<?php
	}

	/**
	 * Renders the log management buttons.
	 *
	 * @since 1.3.0
	 */
	public function log_management_callback() {
		if ( ! class_exists( 'PayStar_Connect_Logger' ) ) {
			require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-logger.php';
		}
		$logger      = new PayStar_Connect_Logger();
		$log_file_path = $logger->get_log_file_path();

		if ( file_exists( $log_file_path ) && filesize( $log_file_path ) > 0 ) :
			$download_url = add_query_arg(
				[
					'action' => 'paystar_download_log',
					'nonce'  => wp_create_nonce( 'paystar_download_log_nonce' ),
				],
				admin_url( 'admin-post.php' )
			);
			$clear_url = add_query_arg(
				[
					'action' => 'paystar_clear_log',
					'nonce'  => wp_create_nonce( 'paystar_clear_log_nonce' ),
				],
				admin_url( 'admin-post.php' )
			);
			?>
			<a href="<?php echo esc_url( $download_url ); ?>" class="button button-primary"><?php esc_html_e( 'دانلود فایل لاگ', 'paystar-connect' ); ?></a>
			<a href="<?php echo esc_url( $clear_url ); ?>" class="button button-secondary" onclick="return confirm('<?php esc_attr_e( 'آیا از پاک کردن فایل لاگ مطمئن هستید؟', 'paystar-connect' ); ?>');"><?php esc_html_e( 'پاک کردن لاگ', 'paystar-connect' ); ?></a>
			<p class="description"><?php esc_html_e( 'فایل لاگ حاوی اطلاعات تراکنش‌ها است. پس از اتمام عیب‌یابی آن را پاک کنید.', 'paystar-connect' ); ?></p>
		<?php else : ?>
			<p class="description"><?php esc_html_e( 'فایل لاگ وجود ندارد یا خالی است. برای ایجاد آن، حالت عیب‌یابی را فعال کرده و یک تراکنش آزمایشی انجام دهید، سپس این صفحه را رفرش کنید.', 'paystar-connect' ); ?></p>
		<?php endif;
	}


	/**
	 * Handles the request to download the log file.
	 *
	 * @since 1.3.0
	 */
	public function handle_download_log() {
		if ( ! isset( $_GET['nonce'] ) || ! wp_verify_nonce( sanitize_key( $_GET['nonce'] ), 'paystar_download_log_nonce' ) ) {
			wp_die( esc_html__( 'لینک معتبر نیست.', 'paystar-connect' ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'شما اجازه دسترسی به این فایل را ندارید.', 'paystar-connect' ) );
		}

		if ( ! class_exists( 'PayStar_Connect_Logger' ) ) {
			require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-logger.php';
		}
		$logger   = new PayStar_Connect_Logger();
		$log_file = $logger->get_log_file_path();

		if ( file_exists( $log_file ) ) {
			header( 'Content-Description: File Transfer' );
			header( 'Content-Type: application/octet-stream' );
			header( 'Content-Disposition: attachment; filename="' . basename( $log_file ) . '"' );
			header( 'Expires: 0' );
			header( 'Cache-Control: must-revalidate' );
			header( 'Pragma: public' );
			header( 'Content-Length: ' . filesize( $log_file ) );
			readfile( $log_file ); // phpcs:ignore
			exit;
		} else {
			wp_die( esc_html__( 'فایل لاگ یافت نشد.', 'paystar-connect' ) );
		}
	}

	/**
	 * Handles the request to clear the log file.
	 *
	 * @since 1.3.0
	 */
	public function handle_clear_log() {
		if ( ! isset( $_GET['nonce'] ) || ! wp_verify_nonce( sanitize_key( $_GET['nonce'] ), 'paystar_clear_log_nonce' ) ) {
			wp_die( esc_html__( 'لینک معتبر نیست.', 'paystar-connect' ) );
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'شما اجازه این کار را ندارید.', 'paystar-connect' ) );
		}

		if ( ! class_exists( 'PayStar_Connect_Logger' ) ) {
			require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-logger.php';
		}
		$logger = new PayStar_Connect_Logger();
		$logger->clear();

		$redirect_url = wp_nonce_url( admin_url( 'admin.php?page=paystar-connect-settings&log-cleared=true' ), 'paystar_log_cleared_notice' );
		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * Displays an admin notice after the log file has been cleared.
	 *
	 * @since 1.3.0
	 */
	public function show_clear_log_notice() {
		if ( isset( $_GET['log-cleared'] ) && 'true' === $_GET['log-cleared'] && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( sanitize_key( $_GET['_wpnonce'] ), 'paystar_log_cleared_notice' ) ) {
			?>
			<div class="notice notice-success is-dismissible">
				<p><?php esc_html_e( 'فایل لاگ با موفقیت پاک شد.', 'paystar-connect' ); ?></p>
			</div>
			<?php
		}
	}
}