<?php
/**
 * Manages the synchronization of data between the website and the Paystar Starshop platform.
 *
 * This class is responsible for scheduling, executing, and monitoring bulk data sync operations.
 * It handles AJAX requests for starting syncs, getting status updates, and canceling operations.
 * It is designed as a singleton to ensure only one instance manages the sync processes.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/includes
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Paystar_Connect_Sync_Manager {
    private static $instance = null;
    private $handlers = [];

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('wp_ajax_starshop_schedule_bulk_sync', [$this, 'ajax_schedule_bulk_sync']);
        add_action('wp_ajax_starshop_get_sync_status', [$this, 'ajax_get_sync_status']);
        add_action('wp_ajax_starshop_cancel_all_syncs', [$this, 'ajax_cancel_all_syncs']);
    }

    public function register_handler($key, $handler_instance) {
        $this->handlers[$key] = $handler_instance;
    }

    public function get_handler($key) {
        return isset($this->handlers[$key]) ? $this->handlers[$key] : null;
    }

    public function get_registered_handlers() {
        return $this->handlers;
    }

    public function ajax_schedule_bulk_sync() {
        check_ajax_referer('starshop_bulk_sync_nonce', 'nonce');
        $sync_type = isset($_POST['sync_type']) ? sanitize_key($_POST['sync_type']) : '';
        $handler   = $this->get_handler($sync_type);

        if ($this->is_processing() && !$this->is_process_stuck()) {
            wp_send_json_error(['message' => 'یک عملیات همگام‌سازی دیگر در حال اجرا است.']);
            return;
        }

        if (!$handler || !function_exists('as_schedule_single_action')) {
            wp_send_json_error(['message' => 'نوع همگام‌سازی نامعتبر است یا زمان‌بند اقدامات فعال نیست.']);
        }

        // Step 1: Forcefully clean up all previous sync data and scheduled actions.
        $this->cleanup_old_sync_runs();
        foreach ($this->get_registered_handlers() as $registered_handler) {
            if (property_exists($registered_handler, 'BATCH_SYNC_ACTION')) {
                as_unschedule_all_actions($registered_handler::BATCH_SYNC_ACTION);
            }
        }

        // Step 2: Create a new run ID and immediately store a clean, empty slate for it.
        // This prevents status checks from fetching stale data from a previous run.
        $run_id = 'paystar_sync_' . $sync_type . '_' . time();
        update_option('starshop_sync_current_run_id', $run_id);
        update_option($run_id, [
            'total'         => 0,
            'processed'     => 0,
            'created'       => 0,
            'updated'       => 0,
            'errors'        => 0,
            'processed_ids' => [],
            'error_logs'    => [],
            'initial_start' => true,
            'sync_type'     => $sync_type,
        ]);

        $all_item_ids    = $handler->get_all_item_ids();
        $unique_item_ids = array_unique($all_item_ids);

        if (empty($unique_item_ids)) {
            wp_send_json_success(['message' => 'هیچ موردی برای همگام‌سازی وجود ندارد!']);
            return;
        }

        // Step 3: Now, update the run data with the actual total count.
        $run_data = [
            'total'         => count($unique_item_ids),
            'processed'     => 0,
            'created'       => 0,
            'updated'       => 0,
            'errors'        => 0,
            'processed_ids' => [],
            'error_logs'    => [],
            'initial_start' => true,
            'sync_type'     => $sync_type,
        ];
        update_option($run_id, $run_data);

        foreach ($unique_item_ids as $item_id) {
            as_schedule_single_action(time(), $handler::BATCH_SYNC_ACTION, ['item_id' => $item_id, 'run_id' => $run_id]);
        }

        wp_send_json_success([
            'message' => count($unique_item_ids) . ' مورد در حال آماده‌سازی برای همگام‌سازی ...',
            'run_id'  => $run_id
        ]);
    }

    public function ajax_get_sync_status() {
        check_ajax_referer('starshop_bulk_sync_nonce', 'nonce');
        $run_id = isset($_POST['run_id']) ? sanitize_text_field(wp_unslash($_POST['run_id'])) : '';

        if (empty($run_id)) {
            wp_send_json_error(['message' => 'شناسه اجرا نامعتبر است. لطفاً صفحه را تازه کنید و دوباره تلاش کنید.']);
            return;
        }

        $run_data = get_option($run_id);
        if (!$run_data) {
            wp_send_json_success(['pending' => 0, 'total' => 0, 'initial_start' => false]);
            return;
        }

        $sync_type = $run_data['sync_type'];
        $handler = $this->get_handler($sync_type);
        if (!$handler || !function_exists('as_get_scheduled_actions')) {
            wp_send_json_success(['pending' => 0, 'total' => 0, 'initial_start' => false]);
            return;
        }

        $pending = count(as_get_scheduled_actions(['hook' => $handler::BATCH_SYNC_ACTION, 'status' => 'pending']));
        
        wp_send_json_success([
            'pending' => $pending,
            'total' => $run_data['total'],
            'processed' => $run_data['processed'],
            'created' => $run_data['created'],
            'updated' => $run_data['updated'],
            'errors' => $run_data['errors'],
            'error_logs' => isset($run_data['error_logs']) ? $run_data['error_logs'] : [],
            'initial_start' => $run_data['initial_start']
        ]);
    }

    public function ajax_cancel_all_syncs() {
        check_ajax_referer('starshop_bulk_sync_nonce', 'nonce');
        $run_id = get_option('starshop_sync_current_run_id');
        if ($run_id) {
            $run_data = get_option($run_id);
            if ($run_data && isset($run_data['sync_type'])) {
                $sync_type = $run_data['sync_type'];
                $handler = $this->get_handler($sync_type);
                if ($handler && function_exists('as_unschedule_all_actions')) {
                    as_unschedule_all_actions($handler::BATCH_SYNC_ACTION);
                }
            }
            delete_option($run_id);
        }
        delete_option('starshop_sync_current_run_id');
        wp_send_json_success(['message' => 'همگام‌سازی سراسری متوقف شد!']);
    }

    public function increment_sync_stat($run_id, $item_id, $stat_type) {
        $run_data = get_option($run_id);
    
        // Stop if no run data or item already processed
        if (!$run_data || in_array($item_id, $run_data['processed_ids'])) {
            return;
        }
    
        // Use a lock to prevent race conditions from multiple processes
        $lock_key = "paystar_sync_lock_{$run_id}";
        if (get_transient($lock_key)) {
            // If locked, wait a bit and retry once.
            sleep(1);
            $run_data = get_option($run_id);
            if (in_array($item_id, $run_data['processed_ids'])) {
                return;
            }
        }
        set_transient($lock_key, true, 10); // Lock for 10 seconds
    
        // Re-fetch data after lock to ensure it's fresh
        $run_data = get_option($run_id);
    
        $run_data['processed']++;
        $run_data['processed_ids'][] = $item_id;
    
        if (in_array($stat_type, ['created', 'updated', 'errors'])) {
            $run_data[$stat_type]++;
        }
    
        update_option($run_id, $run_data);
        delete_transient($lock_key);
    }
    public function add_error_log( $run_id, $item_id, $error_details ) {
        	$run_data = get_option( $run_id );
        	// Safely initialize if run_data is corrupted or missing.
        	if ( ! is_array( $run_data ) ) {
        		$run_data = [];
        	}
       
        	if ( ! isset( $run_data['error_logs'] ) || ! is_array( $run_data['error_logs'] ) ) {
        		$run_data['error_logs'] = [];
        	}
       
        	$error_reason = $error_details['reason'] ?? ( is_string( $error_details ) ? $error_details : 'Unknown Error' );
        	// Normalize reason for more reliable duplicate detection.
        	$normalized_reason = trim( strtolower( $error_reason ) );
        	$unique_error_key = md5( $item_id . '_' . $normalized_reason );
       
        	// Check for duplicates before adding.
        	foreach ( $run_data['error_logs'] as $log ) {
        		// Prioritize checking the unique key.
        		if ( isset( $log['unique_key'] ) && $log['unique_key'] === $unique_error_key ) {
        			return; // Don't add duplicate error for the same item and reason.
        		}
        	}
       
        	$timestamp = time();
        	$log_entry = [
        		'item_id'             => $item_id,
        		'unique_key'          => $unique_error_key,
        		'timestamp'           => $timestamp,
        		'formatted_timestamp' => wp_date( get_option( 'date_format', 'Y-m-d H:i:s' ) . ' ' . get_option( 'time_format', '' ), $timestamp ),
        	];
       
        	if ( is_array( $error_details ) ) {
        		$log_entry = array_merge( $log_entry, $error_details );
        	} else {
        		$log_entry['reason'] = $error_details;
        	}
       
        	$run_data['error_logs'][] = $log_entry;
       
        	update_option( $run_id, $run_data );
        }
       
        public function handle_sync_error( $result, $api_data, $item_id, $run_id, callable $clear_id_callback, callable $update_id_callback, PayStar_Connect_Starshop_Product_API $api_service ) {
            	$settings = get_option( 'paystar_connect_settings' );
            	$store_id = $settings['store_id'] ?? '';
            	if ( ! is_wp_error( $result ) ) {
            		return null; // Not an error, nothing to handle.
            	}
            
            	$get_response_body = function( $error_result ) {
            		if ( ! is_wp_error( $error_result ) ) {
            			return [];
            		}
            		$error_data = $error_result->get_error_data();
            		$body = $error_data['body'] ?? [];
            
            		if ( is_string( $body ) ) {
            			$decoded = json_decode( $body, true );
            			return json_last_error() === JSON_ERROR_NONE ? $decoded : [];
            		}
            		return is_array( $body ) ? $body : [];
            	};
            
            	$response_body = $get_response_body( $result );
            
            	$has_invalid_id_key = isset( $response_body['data']['product_id'] );
            	$has_duplicate_code_key = isset( $response_body['data']['product_code'] );
            
            	// Scenario 1: Invalid product_id key exists. Silently attempt to re-create the product.
            	if ( $has_invalid_id_key ) {
            		$new_result = $api_service->create_product( $api_data );
            
            		// If re-creation is successful, clear old ID, update with new ID, and we're done.
            		if ( ! is_wp_error( $new_result ) && ! empty( $new_result['data']['id'] ) ) {
            			call_user_func( $clear_id_callback ); // Clear old ID only on success.
            			call_user_func( $update_id_callback, $new_result['data']['id'] );
            			$this->increment_sync_stat( $run_id, $item_id, 'created' );
            			return null; // Error resolved.
            		}
            		// If re-creation fails, we fall through to the generic error handler below, using the NEW result.
            		$result = $new_result;
            	}
            	// Re-check for duplicate code after a failed creation attempt.
            	$final_response_body = $get_response_body( $result );
            	$has_invalid_id_key_final = isset( $final_response_body['data']['product_id'] );
            	$has_duplicate_code_key_final = isset( $final_response_body['data']['product_code'] );
            
            	// Scenario 2: ONLY a duplicate product_code key exists.
            	if ( $has_duplicate_code_key_final && ! $has_invalid_id_key_final ) {
            		$product_code = ! empty( $api_data['product_code'] ) ? esc_html( $api_data['product_code'] ) : 'نامشخص';
            		return [
            			'title'    => 'کد محصول تکراری است.',
            			'reason'   => 'این کد محصول قبلا در استارشاپ ثبت شده است.',
            			'solution' => 'در <a href="https://paystar.shop/panel/' . esc_attr( $store_id ) . '/admin/products" target="_blank">پلتفرم استارشاپ</a> محصول با کد <code>' . $product_code . '</code> را بصورت دستی حذف کنید.'
            		];
            	}
            
            	// Generic error for all other unhandled cases.
            	$error_message = is_wp_error( $result ) ? $result->get_error_message() : 'خطای نامشخص';
            	if ( isset( $final_response_body['message'] ) ) {
            		$error_message = $final_response_body['message'];
            	} elseif ( ! empty( $final_response_body ) ) {
            		$details_to_show = $final_response_body['data'] ?? $final_response_body;
            		$error_message = 'پاسخ دریافتی از سرور: ' . wp_json_encode( $details_to_show, JSON_UNESCAPED_UNICODE );
            	}
            
            	return [
            		'title'   => 'خطای همگام‌سازی',
            		'reason'  => $error_message,
            		'solution' => 'لطفاً جزئیات خطا را بررسی کرده و در صورت نیاز با پشتیبانی تماس بگیرید.'
            	];
            }
   
    private function is_processing() {
    	$run_id = get_option( 'starshop_sync_current_run_id' );
    	if ( ! $run_id ) {
            return false;
        }

        $run_data = get_option($run_id);
        if (!$run_data) {
            return false;
        }

        $sync_type = $run_data['sync_type'];
        $handler = $this->get_handler($sync_type);
        if (!$handler || !function_exists('as_get_scheduled_actions')) {
            return false;
        }

        $pending_actions = as_get_scheduled_actions(['hook' => $handler::BATCH_SYNC_ACTION, 'status' => 'pending']);
        $in_progress_actions = as_get_scheduled_actions(['hook' => $handler::BATCH_SYNC_ACTION, 'status' => 'in-progress']);

        return !empty($pending_actions) || !empty($in_progress_actions);
    }

    private function is_process_stuck() {
        $run_id = get_option('starshop_sync_current_run_id');
        if (!$run_id) {
            return false; // No process running
        }
    
        $run_data = get_option($run_id);
        if (empty($run_data) || !isset($run_data['sync_type'])) {
            return false;
        }
    
        $handler = $this->get_handler($run_data['sync_type']);
        if (!$handler || !function_exists('as_get_scheduled_actions')) {
            return false;
        }
    
        $in_progress_actions = as_get_scheduled_actions(['hook' => $handler::BATCH_SYNC_ACTION, 'status' => 'in-progress']);
    
        if (empty($in_progress_actions)) {
            return false; // No actions are currently in progress
        }
    
        $stuck = false;
        foreach ($in_progress_actions as $action) {
            $scheduled_date = $action->get_schedule()->get_date();
            if ($scheduled_date && $scheduled_date->getTimestamp() < (time() - 300)) {
                // If an action has been in-progress for more than 5 minutes, consider it stuck
                $stuck = true;
                break;
            }
        }
    
        return $stuck;
    }
   
    private function cleanup_old_sync_runs() {
        $stale_options = get_transient('paystar_stale_sync_options');
        if (false === $stale_options) {
            $all_options = wp_load_alloptions();
            $stale_options = [];
            foreach ($all_options as $name => $value) {
                if (strpos($name, 'paystar_sync_') === 0) {
                    $stale_options[] = $name;
                }
            }
            set_transient('paystar_stale_sync_options', $stale_options, 5 * MINUTE_IN_SECONDS);
        }

        if ( ! empty( $stale_options ) ) {
            foreach ( $stale_options as $option_name ) {
                delete_option( $option_name );
            }
        }
        delete_transient('paystar_stale_sync_options');
    
        // Delete the pointer to the current run ID.
        delete_option('starshop_sync_current_run_id');
    }
   }