<?php
/**
 * Initializes the Paystar Connect integration for Gravity Forms.
 *
 * This class is responsible for bootstrapping the Gravity Forms integration.
 * It adds the necessary hooks to integrate the Paystar gateway settings into
 * the Gravity Forms UI and loads the main gateway processing class.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/integrations/gravity-forms
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Guardian Clause: Stop loading if Gravity Forms is not active to prevent fatal errors.
if ( ! class_exists( 'GFCommon' ) ) {
    return;
}

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class PayStar_Connect_Gravity_Forms_Integration.
 *
 * Sets up the hooks required to add the Paystar gateway to Gravity Forms.
 */
class PayStar_Connect_Gravity_Forms_Integration {

	/**
	 * Constructor for the Gravity Forms integration.
	 *
	 * Registers all the necessary filters and actions with WordPress and Gravity Forms.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// This filter adds our custom settings panel to the "Form Settings" page for each form.
		add_filter( 'gform_form_settings_fields', [ $this, 'add_gateway_settings_page' ], 10, 2 );

		// The main gateway add-on is loaded on 'gform_loaded' to ensure all GF functions are available.
		// add_action( 'gform_loaded', [ $this, 'load_gateway_addon' ], 5 ); // This was the incorrect approach.

		// Enqueue custom CSS for the payment confirmation messages on the front-end.
		// This hook ensures the stylesheet only loads when a Gravity Form is actually being displayed.
		add_action( 'gform_enqueue_scripts', [ $this, 'enqueue_custom_styles' ], 10, 2 );
		require_once plugin_dir_path(__FILE__) . 'class-gf-starshop-sync-handler.php';
		new GF_Starshop_Sync_Handler();

		// Load and instantiate the main gateway handler to activate payment processing.
		require_once plugin_dir_path( __FILE__ ) . 'class-gf-gateway-paystar.php';
		new PayStar_GF_Gateway_Handler();
	}

	/**
	 * Enqueues the stylesheet for the Paystar Gravity Forms integration.
	 *
	 * This styles the confirmation and error messages displayed to the user
	 * after a payment attempt. It's hooked into 'gform_enqueue_scripts' to
	 * only load on pages where a form is present.
	 *
	 * @since 1.0.0
	 *
	 * @param array $form    The current Gravity Forms form object.
	 * @param bool  $is_ajax Whether the form is being displayed or submitted via AJAX.
	 * @return void
	 */
	public function enqueue_custom_styles( $form, $is_ajax ) {
		// Define the URL to the CSS file.
		$css_url = PAYSTAR_CONNECT_URL . 'integrations/gravity-forms/assets/css/gf-paystar.css';

		// Use the main plugin version for cache-busting.
		$version = PAYSTAR_CONNECT_VERSION;

		// Enqueue the style using a unique handle.
		wp_enqueue_style( 'paystar-gf-styles', $css_url, [], $version );
	}

	/**
	 * Adds the Paystar gateway settings tab and fields to the form settings page.
	 *
	 * This method uses the 'gform_form_settings_fields' filter to add a new "Paystar"
	 * section. It dynamically populates dropdowns with the form's fields, allowing
	 * the admin to map fields like name, email, and phone to the data required by the gateway.
	 *
	 * @since 1.0.0
	 *
	 * @param array $fields The existing array of settings sections from Gravity Forms.
	 * @param array $form   The current form object being edited.
	 * @return array The modified settings array including the new Paystar section.
	 */
	public function add_gateway_settings_page( array $fields, array $form ): array {

		// --- Step 1: Dynamically populate field mapping dropdowns ---
		// Start with a default "Select" option for each dropdown.
		$email_fields = $phone_fields = $name_fields = $lastname_fields = [
			[ 'value' => '', 'label' => '— انتخاب کنید —' ], // "— Select —"
		];

		// Ensure Gravity Forms field classes are available before trying to use them.
		if ( class_exists('GF_Field_Text') ) {


			// Loop through all fields in the current form to find suitable candidates for mapping.
			foreach ( $form['fields'] as $field ) {
				// In this context, $field is a GF_Field object.
				// A field can be used for email if it's an Email type or a generic Text type.
				if ( in_array( $field->type, [ 'email', 'text' ] ) ) {
					$email_fields[] = [ 'value' => $field->id, 'label' => $field->label ];
				}
				// A field can be used for phone if it's a Phone type or a generic Text type.
				if ( in_array( $field->type, [ 'phone', 'text' ] ) ) {
					$phone_fields[] = [ 'value' => $field->id, 'label' => $field->label ];
				}
				// Handle simple text fields for first name and last name.
				if ( $field->type === 'text' ) {
					$name_fields[]     = [ 'value' => $field->id, 'label' => $field->label ];
					$lastname_fields[] = [ 'value' => $field->id, 'label' => $field->label ];
				}

				// Handle complex "Name" fields with sub-inputs (First, Last, etc.).
				if ( $field->type === 'name' && is_array( $field->inputs ) ) {
					foreach ( $field->inputs as $input ) {
						// Skip hidden inputs.
						if ( ! empty( $input['isHidden'] ) ) {
							continue;
						}

						// Identify sub-fields by the decimal part of their ID.
						// This is the most reliable method based on the debug log.
						$id_parts = explode( '.', (string) $input['id'] );
						$sub_id   = end( $id_parts );

						if ( '3' === $sub_id ) { // First Name
							$name_fields[] = [
								'value' => $input['id'],
								'label' => sprintf( '%s ← %s', $field->label, $input['label'] ),
							];
						} elseif ( '6' === $sub_id ) { // Last Name
							$lastname_fields[] = [
								'value' => $input['id'],
								'label' => sprintf( '%s ← %s', $field->label, $input['label'] ),
							];
						}
					}
				}
			}
		}

		// --- Step 2: Define the settings fields using the Gravity Forms Settings API format ---
		$gateway_fields = [
			[
				'name'    => 'paystar_gf_enable',
				'type'    => 'checkbox',
				'label'   => 'فعالسازی پی‌استار کانکت', // "Activate Paystar Gateway"
				'choices' => [
					[
						'name'  => 'paystar_gf_enable',
						'label' => 'پس از ثبت فرم، کاربر به درگاه پرداخت هدایت شود.', // "Redirect user to payment gateway after submission."
					],
				],
			],
		];

		// --- Step 2.5: Conditionally add Starshop Direct notice ---
		$options = get_option( 'paystar_connect_settings' );
		if ( ! empty( $options['base_domain'] ) && 'starshop' === $options['base_domain'] ) {
			$gateway_fields[] = [
				'name' => 'starshop_store_notice',
				'type' => 'html',
				'html' => '<div class="gform-alert" style="display: inline-block !important;">' .
						  '⚠️ <b>' . esc_html__( 'توجه!', 'paystar-connect' ) . '</b><br>' .
						  wp_kses_post( __( 'شما روش <b>«استارشاپ (متصل به فروشگاه)»</b> را انتخاب کرده‌اید.<br>در فرم خود <b>باید یک فیلد <i class="gform-icon gform-icon--name-2"></i>نام</b> به فرم خود اضافه کرده و از قسمت پایین، «نام» و «نام‌خانوادگی» را مطابق فرمِ ساخته شده، مشخص کنید.', 'paystar-connect' ) ) .
						  '</div>',
			];
		}

		$gateway_fields = array_merge(
			$gateway_fields,
			[
				[
					'name'    => 'paystar_gf_fname_field',
					'type'    => 'select',
					'label'   => 'فیلد نام', // "First Name Field"
					'tooltip' => 'فیلدی که نام کوچک کاربر را دریافت می‌کند انتخاب کنید.', // "Select the field that captures the user's first name."
					'choices' => $name_fields,
				],
				[
					'name'    => 'paystar_gf_lname_field',
					'type'    => 'select',
					'label'   => 'فیلد نام خانوادگی', // "Last Name Field"
					'tooltip' => 'فیلدی که نام خانوادگی کاربر را دریافت می‌کند انتخاب کنید.', // "Select the field that captures the user's last name."
					'choices' => $lastname_fields,
				],
				[
					'name'    => 'paystar_gf_phone_field',
					'type'    => 'select',
					'label'   => 'فیلد موبایل', // "Mobile Field"
					'tooltip' => 'فیلدی که شماره موبایل کاربر را دریافت می‌کند انتخاب کنید.', // "Select the field that captures the user's mobile number."
					'choices' => $phone_fields,
				],
				[
					'name'    => 'paystar_gf_email_field',
					'type'    => 'select',
					'label'   => 'فیلد ایمیل', // "Email Field"
					'tooltip' => 'فیلدی که ایمیل کاربر را دریافت می‌کند انتخاب کنید.', // "Select the field that captures the user's email."
					'choices' => $email_fields,
				],
			]
		);

		// --- Step 3: Conditionally add Starshop Sync controls ---
		if (
			!empty($options['base_domain']) && 'starshop' === $options['base_domain'] &&
			!empty($options['store_id']) && !empty($options['store_api_key'])
		) {
			$gateway_fields[] = [
				'name'  => 'starshop_sync_status',
				'type'  => 'html',
				'label' => '<strong style="display: inline-block; margin-top: 15px;">وضعیت در فروشگاه‌ساز استارشاپ</strong>', // "Starshop Sync"
				'html'  => $this->render_starshop_sync_status_field($form),
			];
		}


		// --- Step 4: Create the settings section and add the fields to it ---
		// This adds a new tab titled "Paystar" to the Form Settings area.
		$fields['paystar_gateway'] = [
			'title'  => 'پی‌استار', // "Paystar"
			'fields' => $gateway_fields,
		];

		// Return the final, modified array of settings sections.
		return $fields;
	}

	/**
		* Renders the HTML for the Starshop sync status and button.
		*
		* @param array $form The current form object.
		* @return string The HTML content.
		*/
	public function render_starshop_sync_status_field($form) {
		$starshop_product_id = rgar($form, '_starshop_product_id');

		$status_text = '<span id="starshop-sync-status-text">';
		if (!empty($starshop_product_id)) {
			$status_text .= '✅ ' . esc_html__('همگام‌سازی شده', 'paystar-connect');
			$status_text .= '<br>' . esc_html__('شناسه یکتا: ', 'paystar-connect') . esc_html($starshop_product_id);
			$status_text .= '<br><br><span style="font-size: 12px; font-style: italic;">' . esc_html__('پس از ذخیره فرم، اطلاعات آن بطور خودکار با فروشگاه‌ساز همگام‌سازی می‌شود.', 'paystar-connect') . '</span> ';
		} else {
			$status_text .= '❌ ' . esc_html__('همگام‌سازی نشده', 'paystar-connect');
		}
		$status_text .= '</span>';

		// Return the combined HTML. The nonce is crucial for security.
		return $status_text;
	}
}