<?php
/**
 * Gravity Forms Integration Time-Sensitive Functions.
 *
 * This file contains functions for the Gravity Forms integration that must be loaded early
 * in the WordPress lifecycle. It is included directly by the main integration file,
 * `class-gravity-forms-integration.php`.
 *
 * Its primary responsibility is to register the custom Iranian currencies (IRR and IRT)
 * with Gravity Forms, ensuring they are available for selection in form settings and
 * for processing during payment transactions.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/integrations/Gravity_Forms
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
} 

/**
 * Defines the full properties for Iranian currencies to make them available in Gravity Forms.
 *
 * Gravity Forms requires a complete definition for any currency it uses for pricing fields
 * and payment gateway transactions. This function hooks into `gform_currencies` to add
 * the Iranian Rial (IRR) and Iranian Toman (IRT) to the list of recognized currencies.
 * Without this, selecting these currencies in form settings would not be possible.
 *
 * @since 1.0.0
 *
 * @param array $currencies An associative array of existing currency definitions, where each key
 *                          is the currency code (e.g., 'USD').
 * @return array The modified array of currencies with IRR and IRT definitions added.
 */
function paystar_gf_define_iranian_currencies( $currencies ) {
	// The array structure for each currency is dictated by Gravity Forms.
	// We must provide all the required keys for it to be recognized and formatted correctly.

	// Define Iranian Rial (IRR).
	$currencies['IRR'] = [
		'name'               => 'ریال ایران',
		'symbol_left'        => '',
		'symbol_right'       => 'ریال',
		'symbol_padding'     => ' ',
		'thousand_separator' => ',',
		'decimal_separator'  => '.',
		'decimals'           => 0,
		'code'               => 'IRR',
	];

	// Define Iranian Toman (IRT), which is a common superunit of the Rial.
	$currencies['IRT'] = [
		'name'               => 'تومان ایران',
		'symbol_left'        => '',
		'symbol_right'       => 'تومان',
		'symbol_padding'     => ' ',
		'thousand_separator' => ',',
		'decimal_separator'  => '.',
		'decimals'           => 0,
		'code'               => 'IRT',
	];

	return $currencies;
}
// Register the custom currency definitions with Gravity Forms. This filter is the
// standard way to introduce new currencies to the system.
add_filter( 'gform_currencies', 'paystar_gf_define_iranian_currencies' );


/**
 * Adds Iranian currencies to the "Common Currencies" list in Gravity Forms settings.
 *
 * To improve user experience for administrators, this function hooks into `gform_common_currencies`
 * and adds IRR and IRT to the dropdown list of frequently used currencies. This makes
 * them easier for administrators to find and select when configuring a form's currency,
 * as they won't have to scroll through the full list of world currencies.
 *
 * @since 1.0.0
 *
 * @param array $common_currencies A simple array of common currency codes (e.g., ['USD', 'EUR']).
 * @return array The modified array with 'IRR' and 'IRT' appended.
 */
function paystar_gf_add_iranian_to_common_currencies( $common_currencies ) {
	// Add our custom currencies to the list for quick access in the admin UI.
	$common_currencies[] = 'IRR';
	$common_currencies[] = 'IRT';

	return $common_currencies;
}
// Register the function to modify the common currencies list. This enhances usability
// for site administrators setting up forms.
add_filter( 'gform_common_currencies', 'paystar_gf_add_iranian_to_common_currencies' );