<?php
/**
 * Paystar Payment Gateway for WooCommerce.
 *
 * This class is the actual payment gateway that integrates with WooCommerce's systems.
 * It extends the WC_Payment_Gateway class and is responsible for defining settings,
 * processing payments at checkout, and checking for availability.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/integrations/woocommerce
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Guardian Clause: Stop loading if WooCommerce is not active to prevent fatal errors.
if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
    return;
}


// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Inherits from WC_Payment_Gateway to provide the Paystar payment method.
 */
class WC_Gateway_PayStar extends WC_Payment_Gateway {

	private $logger;

	/**
	 * Constructor for the gateway.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		if ( ! class_exists( 'PayStar_Connect_Logger' ) ) {
			require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-logger.php';
		}
		$this->logger = new PayStar_Connect_Logger();		
		// Basic gateway properties
		$this->id         = 'paystar_connect';
		$this->has_fields = false; // No custom fields on the checkout page itself
		$this->supports   = [ 'products', 'blocks' ]; // Supports products and WooCommerce Blocks
		$this->icon       = PAYSTAR_CONNECT_URL . 'integrations/woocommerce/assets/images/psc-logo.png';

		// Load the settings fields and initialize the settings values.
		$this->init_form_fields();
		$this->init_settings();

		// Define user-facing properties from the settings.
		$this->method_title       = 'پی‌استار کانکت';
		$this->method_description = 'پرداخت توسط افزونه پرداخت پی‌استار کانکت';
		$this->title              = $this->get_option( 'title' );
		$this->description        = $this->get_option( 'description' );

		// Hook to save the settings when the admin clicks "Save changes".
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, [ $this, 'process_admin_options' ] );

		// Dynamically set the icon on the checkout page.
		add_filter( 'woocommerce_gateway_icon', [ $this, 'get_checkout_icon' ], 10, 2 );
	}

	/**
	 * Defines the settings fields for the gateway in the WooCommerce admin area.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function init_form_fields() {
		$main_options = get_option( 'paystar_connect_settings' );
		$base_domain  = $main_options['base_domain'] ?? 'paystar';

		$default_title = 'پرداخت آنلاین پی‌استار';
		if ( in_array( $base_domain, [ 'starshop_direct', 'starshop' ] ) ) {
			$default_title = 'پرداخت آنلاین';
		}

		$this->form_fields = [
			'main_config_title'     => [
				'title' => 'فعالسازی', // "Activation"
				'type'  => 'title',
				'id'    => 'woocommerce_paystar_connect_main_config_title',
			],
			'enabled'               => [
				'title'   => 'فعال/غیرفعال', // "Enable/Disable"
				'type'    => 'checkbox',
				'label'   => 'فعالسازی پرداخت با پی‌استار کانکت', // "Activate Paystar Gateway"
				'default' => 'yes',
			],

			'display_options_title' => [
				'title' => 'جزئیات نمایش در صفحهٔ تسویه حساب', // "Display Details on Checkout Page"
				'type'  => 'title',
				'id'    => 'woocommerce_paystar_connect_display_options_title',
			],
			'title'                 => [
				'title'       => 'عنوان درگاه', // "Gateway Title"
				'type'        => 'text',
				'description' => 'این عنوان در صفحه تسویه حساب به کاربر نمایش داده می‌شود.', // "This title is shown to the user on the checkout page."
				'default'     => $default_title,
				'desc_tip'    => true,
			],
			'description'           => [
				'title'       => 'توضیحات درگاه', // "Gateway Description"
				'type'        => 'textarea',
				'description' => 'توضیحاتی که کاربر در حین انتخاب این درگاه مشاهده می‌کند.', // "Description the user sees when selecting this gateway."
				'default'     => 'پرداخت امن بانکی از طریق کارت‌های متصل به شبکه شتاب',
			],
		];
	}

	/**
	 * Processes the payment when the user clicks "Place Order".
	 *
	 * This method calls the Paystar API to create a transaction and returns
	 * a redirect URL for WooCommerce to send the user to the payment page.
	 *
	 * @since 1.0.0
	 *
	 * @param  int $order_id The ID of the order being processed.
	 * @return array An array containing the result and redirect URL.
	 */
	public function process_payment($order_id) {
		// Step 1: Get the order object as the single source of truth.
		$order = wc_get_order($order_id);

		// Initial check to ensure the order exists.
		if ( ! $order ) {
			$this->logger->log( 'WooCommerce Error: Could not retrieve order object for order ID: ' . $order_id );
			wc_add_notice( 'خطایی هنگام دریافت اطلاعات سفارش رخ داد. سفارش یافت نشد.', 'error' );
			return [ 'result' => 'failure' ];
		}

		// Step 2: Optimistic locking by checking the order status.
		// This critical check prevents reprocessing of completed or canceled orders.
		if ( ! $order->has_status( 'pending' ) ) {
			wc_add_notice( 'این سفارش دیگر برای پرداخت معتبر نیست. لطفاً یک سفارش جدید ثبت کنید.', 'error' );
			return [ 'result' => 'failure' ];
		}

		// Step 3: Extract all required data from the $order object.
		// From now on, we only use these local variables.
		$order_id_from_object = $order->get_id();
		$this->logger->log( '--------------------------------' );
		$this->logger->log( 'WooCommerce: Initiating payment process for Order ID: ' . $order_id_from_object );
		$amount               = paystar_connect_get_amount_by_currency( $order->get_total(), $order->get_currency() );
		
		// Build the callback URL using the ID extracted from the object.
		$callback_base         = WC()->api_request_url( 'paystar_connect_callback' );
		$callback_args         = [ 'wc_order_id' => $order_id_from_object ];
		$callback_url_without_nonce = add_query_arg( $callback_args, $callback_base );
		$callback                   = str_replace( '&amp;', '&', $callback_url_without_nonce );

		// Step 4: Conditionally prepare extra data only if required by the service provider.
		$main_options = get_option( 'paystar_connect_settings' );
		$base_domain   = $main_options['base_domain'] ?? 'paystar'; // Default to 'paystar' if not set.
		$extra_data   = [];

		if ( 'starshop' === $base_domain ) {
			// Prepare products for the API.
			$products_for_api = [];
			foreach ( $order->get_items() as $item ) {
				$product = $item->get_product();
				if ( $product ) {
					$products_for_api[] = [
						'code'     => str_pad( (string) $product->get_id(), 2, '0', STR_PAD_LEFT ),
						'price'    => (string) ( round( $item->get_subtotal() ) * 10 ), // Price in Rials
						'quantity' => $item->get_quantity(),
					];
				}
			}

			// Prepare customer data.
			$first_name = $order->get_billing_first_name() ?: 'کاربر';
			$last_name  = $order->get_billing_last_name() ?: 'مهمان';

			$extra_data = [
				'first_name' => $first_name,
				'last_name'  => $last_name,
				'products'   => $products_for_api,
			];
		}

		// Step 5: Call the API with unified data.
		$api    = new PayStar_Connect_API();
		$result = $api->create( $amount, (string) $order_id_from_object, $callback, $extra_data );

		if ( ! empty( $result['status'] ) && true === $result['status'] ) {
			// Transaction created successfully.
			$payment_url = $result['payment_url'] . '?token=' . $result['token'];

			return [
				'result'   => 'success',
				'redirect' => $payment_url,
			];
		} else {
			// Error in creating the transaction.
			$error_message = $result['message'] ?? __( 'یک خطای ناشناخته رخ داد.', 'paystar-connect' );
			$this->logger->log( 'WooCommerce: Payment creation failed for Order ID: ' . $order_id_from_object . '. Reason: ' . $error_message );
			wc_add_notice( __( 'خطا در اتصال به درگاه پرداخت:', 'paystar-connect' ) . ' ' . $error_message, 'error' );
			return [ 'result' => 'failure' ];
		}
	}

	/**
	 * Checks if the payment gateway should be available to the user.
	 *
	 * The gateway is only available if it's enabled and the required API
	 * credentials (Gateway ID and Sign Key) are present in the main plugin settings.
	 *
	 * @since 1.0.0
	 *
	 * @return bool Whether the gateway is available.
	 */
	public function is_available() {
		// First, check if the gateway is enabled in WooCommerce settings.
		if ( ! parent::is_available() ) {
			return false;
		}

		// Retrieve the main plugin settings.
		$main_options = get_option( 'paystar_connect_settings' );

		// The gateway is only available if the Gateway ID and Sign Key are set.
		if ( empty( $main_options['gateway_id'] ) || empty( $main_options['sign_key'] ) ) {
			return false;
		}

		// If all checks pass, the gateway is available for use.
		return true;
	}
	// The starshop_post_payment_sync method has been removed as its logic is now inside the callback_handler.

	/**
	 * Returns the checkout icon HTML.
	 *
	 * @since 1.0.0
	 * @param string $icon The icon HTML.
	 * @param string $id The gateway ID.
	 * @return string The icon HTML.
	 */
	public function get_checkout_icon( $icon, $id ) {
		if ( $this->id !== $id ) {
			return $icon;
		}

		$main_options = get_option( 'paystar_connect_settings' );
		$base_domain  = $main_options['base_domain'] ?? 'paystar';

		$icon_url = PAYSTAR_CONNECT_URL . 'integrations/woocommerce/assets/images/paystar-connect-wc-icon.png';
		if ( in_array( $base_domain, [ 'starshop_direct', 'starshop' ] ) ) {
			$icon_url = PAYSTAR_CONNECT_URL . 'integrations/woocommerce/assets/images/icon-card.png';
		}

		return '<img src="' . esc_url( $icon_url ) . '" alt="' . esc_attr( $this->get_title() ) . '" />';
	}
}