<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class WC_Starshop_Sync_Handler implements Paystar_Connect_Sync_Handler_Interface {

	public const BATCH_SYNC_ACTION = 'paystar_connect_sync_wc_product';
	private      $api_service;
	   private      $sync_manager;
	
	public function __construct() {
		$options = get_option( 'paystar_connect_settings' );
	
		if (
			! empty( $options['base_domain'] ) && 'starshop' === $options['base_domain'] &&
			! empty( $options['store_id'] ) && ! empty( $options['store_api_key'] )
		) {
			$this->api_service = new PayStar_Connect_Starshop_Product_API();
	           $this->sync_manager = Paystar_Connect_Sync_Manager::get_instance();
			add_action( 'save_post_product', [ $this, 'handle_product_save' ], 10, 2 );
			add_action( 'add_meta_boxes_product', [ $this, 'add_sync_meta_box' ] );
			// Hook for the background processor
			add_action( self::BATCH_SYNC_ACTION, [ $this, 'process_sync_action' ], 10, 1 );
			add_filter( 'woocommerce_duplicate_product_exclude_meta', [ $this, 'exclude_starshop_meta_on_duplicate' ] );
		}
	}

    public function handle_product_save($post_id, $post) {
        if ((defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) || wp_is_post_revision($post_id)) {
            return;
        }
    
        $product = wc_get_product($post_id);
        if (!$product) {
            return;
        }
    
        $product_code = 'WC_' . $product->get_id();
        $api_data = [
            'title'        => $product->get_name(),
            'product_code' => $product_code,
            'status'       => $product->get_status() === 'publish' ? '1' : '0',
            'description'  => $product->get_description(),
            'price_type'   => $product->is_type('variable') ? '2' : '1',
            'price'        => $product->is_type('variable') ? '' : $product->get_price(),
            'unlimited'    => !$product->managing_stock() ? '1' : '0',
            'quantity'     => $product->managing_stock() ? (int) $product->get_stock_quantity() : '0',
        ];
    
        $starshop_product_id = get_post_meta($post_id, '_starshop_product_id', true);
        $run_id = get_option('starshop_sync_current_run_id');
        $result = null;
    
        if (!empty($starshop_product_id)) {
            $result = $this->api_service->update_product($starshop_product_id, $api_data);
        } else {
            $result = $this->api_service->create_product($api_data);
        }
    
        if (is_wp_error($result)) {
            $clear_id_callback = function () use ($post_id) {
                delete_post_meta($post_id, '_starshop_product_id');
            };

            $update_id_callback = function ($new_id) use ($post_id) {
                update_post_meta($post_id, '_starshop_product_id', $new_id);
            };

            $error_details = $this->sync_manager->handle_sync_error(
                $result,
                $api_data,
                $post_id,
                $run_id,
                $clear_id_callback,
                $update_id_callback,
                $this->api_service
            );

            if ($error_details) {
            	$log_message = sprintf(
            		'Starshop Sync Error for product %d: %s - %s (Solution: %s)',
            		$post_id,
            		$error_details['title'],
            		$error_details['reason'],
            		$error_details['solution']
            	);
                if ($run_id) {
                    $this->sync_manager->add_error_log($run_id, $post_id, $error_details);
                    $this->sync_manager->increment_sync_stat($run_id, $post_id, 'errors');
                }
            }
        } else {
            if (!empty($starshop_product_id)) {
                $this->sync_manager->increment_sync_stat($run_id, $post_id, 'updated');
            } elseif (!empty($result['data']['id'])) {
                update_post_meta($post_id, '_starshop_product_id', $result['data']['id']);
                $this->sync_manager->increment_sync_stat($run_id, $post_id, 'created');
            }
        }
    }
    public function add_sync_meta_box() {
        add_meta_box(
            'starshop_sync_status_meta_box', // ID
            __('همگام‌سازی با استارشاپ', 'paystar-connect'), // Title
            [$this, 'render_meta_box_content'], // Callback
            'product', // Screen
            'side', // Context
            'high' // Priority
        );
    }

    public function render_meta_box_content($post) {
        wp_nonce_field('starshop_manual_sync_nonce', 'starshop_manual_sync_nonce');
        // Get the Starshop product ID from post meta
        $starshop_product_id = get_post_meta($post->ID, '_starshop_product_id', true);

        if (!empty($starshop_product_id)) {
            // Product is synced
            echo '<p>';
            echo '✅ ' . esc_html__('همگام‌سازی شده', 'paystar-connect');
            echo '<br>' . esc_html__('شناسه یکتا:', 'paystar-connect') . '<strong>' . esc_html($starshop_product_id);
            echo '</strong></p>';
        } else {
            // Product is not synced
            echo '<p>';
            echo '⚠️ ' . esc_html__('همگام‌سازی نشده', 'paystar-connect');
            echo '<br>' . esc_html__('ذخیره محصول بطور خودکار آن را با پلتفرم استارشاپ همگام‌سازی می‌کند.', 'paystar-connect');
            echo '</p>';
        }

        echo '<p class="description" style="font-size: 12px; font-style: italic;"><i>' . esc_html__('پس از ذخیره/به‌روزرسانی محصول، اطلاعات آن بطور خودکار با فروشگاه‌ساز همگام‌سازی می‌شود.', 'paystar-connect') . '</i></p>';
    }
 public function get_id() {
 	return 'woocommerce';
 }

 public function get_name() {
 	return __( 'ووکامرس', 'paystar-connect' );
 }

 public function get_all_item_ids() {
 	$query = new WC_Product_Query( [
 		'limit'  => -1,
 		'status' => 'any',
 		'return' => 'ids'
 	] );

 	return $query->get_products();
 }

 public function process_sync_action( $args ) {
    $item_id = is_array($args) && isset($args['item_id']) ? $args['item_id'] : $args;
 	$post = get_post( $item_id );
 	if ( $post && 'product' === $post->post_type ) {
 		// Reuse the existing save handler logic
 		$this->handle_product_save( $item_id, $post );
 	}
 }

 public function process_batch( $batch ) {
 $processed_count = 0;
 foreach ( $batch as $item_id ) {
  $this->process_sync_action( $item_id );
  $processed_count++;
 }
 return [
 	'success'         => true,
 	'processed_count' => $processed_count,
 ];
 }

 /**
  * Exclude Starshop meta data from being duplicated.
  *
  * @param array $meta_keys A list of meta keys to exclude from duplication.
  *
  * @return array
  */
 public function exclude_starshop_meta_on_duplicate( $meta_keys ) {
 	$meta_keys[] = '_starshop_product_id';

 	return $meta_keys;
 }
}