<?php
/**
 * Handles all API communications with the PayStar.shop Product/Service API.
 *
 * @package    PayStar_Connect
 * @subpackage PayStar_Connect/includes
 * @author     PayStar
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Final class PayStar_Connect_Starshop_Product_API.
 */
final class PayStar_Connect_Starshop_Product_API {

	/**
	 * To store the Starshop Store API Key.
	 *
	 * @var string
	 */
	private $store_api_key;

	/**
	 * To store the Starshop Store ID.
	 *
	 * @var string
	 */
	private $store_id;
	private $refresh_token;
	private $options;

	/**
	 * The base endpoint for the product API.
	 */
	private const API_BASE_URL = 'https://coreshop.paystar.shop/api/product/service/';
	private const REFRESH_ENDPOINT = 'https://coreshop.paystar.shop/api/api-key/refresh';

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->options = get_option( 'paystar_connect_settings' );
		$this->store_api_key = $this->options['store_api_key'] ?? '';
		$this->store_id      = $this->options['store_id'] ?? '';
		$this->refresh_token = $this->options['store_refresh_token'] ?? '';
	}

	/**
	 * Sends a request to create a new product on Starshop.
	 *
	 * @param array $api_data Product data formatted for the API.
	 * @return array|WP_Error The decoded JSON response from the API or a WP_Error.
	 */
	public function create_product( array $api_data ) {
		return $this->remote_request( 'create', $api_data );
	}

	/**
	 * Sends a request to update an existing product on Starshop.
	 *
	 * @param string $starshop_product_id The existing product ID on Starshop.
	 * @param array  $api_data Product data formatted for the API.
	 * @return array|WP_Error The decoded JSON response from the API or a WP_Error.
	 */
	public function update_product( string $starshop_product_id, array $api_data ) {
		$api_data['product_id'] = $starshop_product_id;
		return $this->remote_request( 'update', $api_data );
	}

	/**
	 * Fetches a product from Starshop using its product code.
	 *
	 * @param string $product_code The product code to search for.
	 * @return array|WP_Error The decoded JSON response or a WP_Error.
	 */
	public function get_product_by_code( string $product_code ) {
		$body = [
			'product_code' => $product_code,
			'store_id'     => $this->store_id,
		];
		return $this->remote_request( 'get-by-code', $body, 'GET' );
	}

	/**
	 * Refreshes the API key using the refresh token.
	 *
	 * This method sends a request to the refresh endpoint to obtain a new API key
	 * and refresh token. If successful, it updates the stored settings.
	 *
	 * @since 1.2.0
	 * @return array On success, returns ['status' => true, 'message' => '...'].
	 *               On failure, returns ['status' => false, 'message' => '...'].
	 */
	public function refresh_api_key() {
		if ( empty( $this->refresh_token ) ) {
			return [
				'status'  => false,
				'message' => 'Refresh token is not set. Cannot refresh API key.',
			];
		}

		$body = [
			'refresh_token' => $this->refresh_token,
		];

		$response = wp_remote_post(
			self::REFRESH_ENDPOINT,
			[
				'method'  => 'POST',
				'headers' => [
					'Content-Type' => 'application/json',
					'Accept'       => 'application/json',
				],
				'body'    => json_encode( $body ),
				'timeout' => 30,
			]
		);

		if ( is_wp_error( $response ) ) {
			return [
				'status'  => false,
				'message' => 'Failed to connect to the refresh service. ' . $response->get_error_message(),
			];
		}

		$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			return [
				'status'  => false,
				'message' => 'Failed to decode JSON response from the refresh service. Error: ' . json_last_error_msg(),
			];
		}

		if ( isset( $response_body['status'] ) && $response_body['status'] === 'ok' && ! empty( $response_body['data']['api_key'] ) ) {
			$this->options['store_api_key']       = $response_body['data']['api_key'];
			$this->options['store_refresh_token'] = $response_body['data']['refresh_token'];
			update_option( 'paystar_connect_settings', $this->options );

			// Update the object's properties with the new values
			$this->store_api_key = $this->options['store_api_key'];
			$this->refresh_token = $this->options['store_refresh_token'];

			return [
				'status'  => true,
				'message' => $response_body['message'] ?? 'API key refreshed successfully.',
			];
		}


		return [
			'status'  => false,
			'message' => $response_body['message'] ?? 'An unknown error occurred during API key refresh.',
		];
	}

	/**
	 * A wrapper for WordPress HTTP API to handle requests to the Starshop Product API.
	 *
	 * @param string $endpoint The API endpoint (e.g., 'create' or 'update').
	 * @param array  $body The request body.
	 * @param string $method The HTTP method ('POST' or 'GET').
	 * @return array|WP_Error The response from the request.
	 */
	private function remote_request( string $endpoint, array $body, string $method = 'POST' ) {
	       if (isset($body['description'])) {
	           $body['description'] = mb_substr($body['description'], 0, 250);
	       }

		$url = self::API_BASE_URL . $endpoint;
		$retries = 3;
		$retry_delay = 5; // seconds

		$args = array(
			'method'  => $method,
			'timeout' => 30,
			'headers' => array(
				'Authorization' => 'Bearer ' . $this->store_api_key,
				'Accept'        => 'application/json',
			),
		);

		for ($i = 0; $i < $retries; $i++) {
			if ( 'POST' === $method ) {
				$args['body'] = array_merge( $body, [ 'store_id' => $this->store_id ] );
				$response     = wp_remote_post( $url, $args );
			} else {
				$url_with_args = add_query_arg( $body, $url );
				$response = wp_remote_get( $url_with_args, $args );
			}

			if ( !is_wp_error( $response ) ) {
				// If the request was successful, break out of the loop
				break;
			}
			
			// If it's a cURL error 28 (timeout), wait and retry
			if ($response->get_error_code() === 'http_request_failed' && strpos($response->get_error_message(), 'cURL error 28') !== false) {
				sleep($retry_delay);
				continue;
			} else {
				// For other errors, don't retry
				return $response;
			}
		}


		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $response_code && 201 !== $response_code ) {
			return $this->handle_api_error( $response, $body );
		}

		return json_decode( wp_remote_retrieve_body( $response ), true );
	}

	/**
		* Handles API errors and formats them for user display.
		*
		* @param array|WP_Error $response The response object from the API call.
		* @param array          $body The original request body.
		* @return WP_Error A formatted error object.
		*/
	private function handle_api_error( $response, $body ) {
		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$decoded_body  = json_decode( $response_body, true );
		$error_message = __( 'An unknown API error occurred.', 'paystar-connect' );

		if ( json_last_error() === JSON_ERROR_NONE && ! empty( $decoded_body['message'] ) ) {
			$error_message = $decoded_body['message'];
			if ( ! empty( $decoded_body['data'] ) && is_array( $decoded_body['data'] ) ) {
				$details = [];
				foreach ( $decoded_body['data'] as $field => $errors ) {
					if ( is_array( $errors ) ) {
						$details[] = $field . ': ' . implode( ', ', $errors );
					}
				}
				if ( ! empty( $details ) ) {
					$error_message .= ' (' . implode( '; ', $details ) . ')';
				}
			}
		}

		return new WP_Error(
			'api_error',
			$error_message,
			[
				'status' => $response_code,
				'body'   => $decoded_body,
			]
		);
	}
}