<?php
/**
 * Paystar Connect Logger
 *
 * This file is responsible for handling all logging operations for the Paystar
 * Connect plugin. It provides a centralized way to manage debug logs.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/includes
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.3.0
 */

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class PayStar_Connect_Logger
 *
 * Handles all logging-related functionalities.
 *
 * @since 1.3.0
 */
class PayStar_Connect_Logger {

	/**
	 * The path to the log file.
	 *
	 * @since  1.3.0
	 * @access private
	 * @var    string
	 */
	private $log_file_path;

	/**
	 * The URL to the log file.
	 *
	 * @since  1.3.0
	 * @access private
	 * @var    string
	 */
	private $log_file_url;

	/**
	 * Constructor for the logger class.
	 *
	 * Sets up the log file path and URL.
	 *
	 * @since 1.3.0
	 */
	public function __construct() {
		$upload_dir        = wp_upload_dir();
		$log_dir           = $upload_dir['basedir'] . '/paystar-connect-logs';
		$this->log_file_path = $log_dir . '/paystar-connect-debug.log';
		$this->log_file_url  = $upload_dir['baseurl'] . '/paystar-connect-logs/paystar-connect-debug.log';

		// Create the log directory if it doesn't exist.
		if ( ! is_dir( $log_dir ) ) {
			wp_mkdir_p( $log_dir );
		}
	}

	/**
	 * Checks if debugging is enabled in the main settings.
	 *
	 * @since  1.3.0
	 * @return bool True if debugging is enabled, false otherwise.
	 */
	public static function is_debugging_enabled() {
		$options = get_option( 'paystar_connect_settings' );
		return ! empty( $options['debug_mode'] ) && 'yes' === $options['debug_mode'];
	}

	/**
	 * Logs a message to the file if debugging is enabled.
	 *
	 * @since 1.3.0
	 * @param string $message The message to log.
	 */
	public function log( $message ) {
		if ( ! self::is_debugging_enabled() ) {
			return;
		}

		$formatted_message = sprintf(
			"[%s] %s\n",
			wp_date( 'Y-m-d H:i:s' ),
			$message
		);

		// Use file_put_contents with LOCK_EX for safe writing.
		file_put_contents( $this->log_file_path, $formatted_message, FILE_APPEND | LOCK_EX ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_file_put_contents
	}

	/**
	 * Clears the log file.
	 *
	 * @since  1.3.0
	 * @return bool True on success, false on failure.
	 */
	public function clear() {
		if ( file_exists( $this->log_file_path ) ) {
			return wp_delete_file( $this->log_file_path );
		}
		return true;
	}

	/**
	 * Gets the path to the log file.
	 *
	 * @since  1.3.0
	 * @return string The log file path.
	 */
	public function get_log_file_path() {
		return $this->log_file_path;
	}

	/**
	 * Gets the URL of the log file.
	 *
	 * @since  1.3.0
	 * @return string The log file URL.
	 */
	public function get_log_file_url() {
		return $this->log_file_url;
	}
}