<?php

/**
 * The core plugin class for Paystar Connect.
 *
 * This is the main class that initializes the plugin, sets up hooks,
 * and loads the necessary integration files for supported plugins.
 * It follows a singleton pattern to ensure it's only loaded once.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/includes
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
} 

class PayStar_Connect {

	/**
	 * The single instance of the class.
	 *
	 * @since 1.0.0
	 * @var   PayStar_Connect|null The single instance of the class.
	 */
	protected static $_instance = null;

	/**
	 * Main PayStar_Connect Instance.
	 *
	 * Ensures only one instance of PayStar_Connect is loaded or can be loaded.
	 * This is the public-facing entry point to the singleton class.
	 *
	 * @since  1.0.0
	 * @static
	 * @return PayStar_Connect Main instance.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * PayStar_Connect Constructor.
	 *
	 * Kicks off the plugin by including core files and initializing hooks.
	 * This is set to protected to prevent direct object creation, enforcing the singleton pattern.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->handle_plugin_update();
		$this->includes();
		$this->init_hooks();
	}

	/**
	 * Handles plugin updates by comparing the current version with the stored version.
	 * If they differ, it means the plugin has been updated, and we can clear caches.
	 *
	 * @since 2.11.0
	 */
	private function handle_plugin_update() {
		$current_version = PAYSTAR_CONNECT_VERSION;
		// Use a specific option name for this plugin's version.
		$stored_version = get_option( 'paystar_connect_db_version', '0.0.0' );

		// If the current version is greater than the stored version, an update has occurred.
		if ( version_compare( $current_version, $stored_version, '>' ) ) {
			// 1. Clear the remote notice transient to force a fresh check.
			// I'm using the corrected transient name you mentioned.
			delete_transient( 'dinor_plugins_remote_notice' );

			// 2. Update the stored version in the database to the current version.
			update_option( 'paystar_connect_db_version', $current_version );
		}
	}

	/**
	 * Include required core files.
	 *
	 * Loads the essential classes needed for the plugin to function,
	 * such as the API handler and the settings page generator.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function includes() {
		// Load the API class for handling communication with the Paystar gateway.
		require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-api.php';
		// Load the Settings class to create and manage the admin settings page.
		require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-settings.php';
		require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-starshop-product-api.php';
		
		// Load the new sync system files.
		require_once PAYSTAR_CONNECT_PATH . 'includes/interfaces/interface-paystar-connect-sync-handler.php';
		require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-sync-manager.php';
	}

	/**
	 * Initialize WordPress hooks.
	 *
	 * Registers all of the hooks and filters used by the core plugin.
	 * This includes setting up the admin menu, enqueueing scripts,
	 * and loading the integration modules at the correct time.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function init_hooks() {
		// Initialize the settings page by instantiating the settings class.
		// This class is responsible for adding the menu item and rendering the settings page.
		new PayStar_Connect_Settings();

		// Hook into WordPress to add admin-specific styles and menu.
		add_action( 'admin_menu', [ $this, 'add_plugin_page' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_styles' ] );

		$this->define_public_hooks();

		// Load integrations for other plugins after all plugins are loaded.
		// We use a high priority (9) to ensure our integration hooks, especially
		// for payment gateways, are registered before other plugins might interfere.
		// Initialize the sync manager before loading integrations.
		$this->init_sync_manager();

		// Load integrations for other plugins after all plugins are loaded.
		// We use a high priority (9) to ensure our integration hooks, especially
		// for payment gateways, are registered before other plugins might interfere.
		add_action( 'plugins_loaded', [ $this, 'load_integrations' ], 9 );

		// Hook for the daily API key refresh.
		add_action( 'paystar_connect_refresh_api_key_cron', [ $this, 'run_api_key_refresh' ] );

		// Hook for displaying remote admin notices.
		add_action( 'admin_notices', [ $this, 'dinor_display_remote_admin_notice' ] );
	}

	/**
		* Executes the API key refresh process.
		*
		* This method is triggered by the scheduled cron job. It instantiates the
		* API class and calls the refresh method.
		*
		* @since 1.2.0
		*/
	public function run_api_key_refresh() {
		// This now correctly points to the Starshop API class.
		$api = new PayStar_Connect_Starshop_Product_API();
		$result = $api->refresh_api_key();

		if ( ! $result['status'] && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		}
	}

	/**
	 * Initializes the Sync Manager and registers handlers.
	 *
	 * @since 2.2.0
	 */
	public function init_sync_manager() {
		$sync_manager = Paystar_Connect_Sync_Manager::get_instance();

		// Make the manager globally accessible.
		$GLOBALS['paystar_connect_sync_manager'] = $sync_manager;
	}

	/**
	 * Load integration files for active plugins.
	 *
	 * This method serves as the core of the modular architecture. It checks
	 * for the presence of supported plugins (e.g., WooCommerce, Gravity Forms)
	 * and loads the corresponding integration class if the plugin is active.
	 *
	 * @since 1.0.0
	 */
	public function load_integrations() {
		$integrations_path = PAYSTAR_CONNECT_PATH . 'integrations/';

		// Check if WooCommerce is active by looking for its main class.
		if ( class_exists( 'WooCommerce' ) ) {
			// If active, load the WooCommerce integration file.
			require_once $integrations_path . 'woocommerce/class-woocommerce-integration.php';
			// Instantiate the integration class to fire its constructor and register its hooks.
			new WooCommerce_Integration();

			// Register the WC sync handler.
			require_once $integrations_path . 'woocommerce/class-wc-starshop-sync-handler.php';
			$GLOBALS['paystar_connect_sync_manager']->register_handler( 'woocommerce', new WC_Starshop_Sync_Handler() );
		}

		// Check if Gravity Forms is active by looking for one of its core classes.
		if ( class_exists( 'GFCommon' ) ) {
			// If active, load the Gravity Forms integration file.
			require_once $integrations_path . 'gravity-forms/class-gravity-forms-integration.php';
			// Instantiate the integration class to fire its constructor and register its hooks.
			new PayStar_Connect_Gravity_Forms_Integration();

			// Register the GF sync handler.
			$GLOBALS['paystar_connect_sync_manager']->register_handler( 'gf-forms', new GF_Starshop_Sync_Handler() );
		}
	}

	/**
		* Defines hooks that run on the public-facing side of the site.
		*
		* @since 1.5.6
		* @access private
		*/
	private function define_public_hooks() {
		add_action( 'wp', [ $this, 'prevent_caching_on_critical_pages' ] );
	}

	/**
		* Prevents caching on critical payment pages.
		*
		* This function hooks into 'wp' and checks if the current page is a checkout page
		* or a payment gateway callback. If so, it defines the DONOTCACHEPAGE constant
		* and sends no-cache headers to prevent issues with outdated nonces.
		*
		* @since 1.5.6
		*/
	public function prevent_caching_on_critical_pages() {
		// Check for WooCommerce pages (checkout and API callbacks).
		if ( function_exists( 'is_checkout' ) && is_checkout() ) {
			if ( ! defined( 'DONOTCACHEPAGE' ) ) {
				define( 'DONOTCACHEPAGE', true );
			}
			nocache_headers();
			return;
		}
	
		if ( class_exists( 'WooCommerce' ) && isset( $_SERVER['REQUEST_URI'] ) ) {
			$request_uri = esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) );
			if ( strpos( $request_uri, 'paystar_callback' ) !== false ) {
				if ( function_exists( 'WC' ) && WC()->is_wc_api_request() ) {
					if ( ! defined( 'DONOTCACHEPAGE' ) ) {
						define( 'DONOTCACHEPAGE', true );
					}
					nocache_headers();
					return;
				}
			}
		}
	
		// Check for Gravity Forms callback.
		// This is a more generic check as GF doesn't have a conditional tag like is_checkout().
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['paystar_gf_callback'] ) ) {
			if ( ! defined( 'DONOTCACHEPAGE' ) ) {
				define( 'DONOTCACHEPAGE', true );
			}
			nocache_headers();
			return;
		}
	}

	/**
	 * Enqueue admin-specific stylesheets.
	 *
	 * This function is hooked into 'admin_enqueue_scripts'. It ensures that
	 * the plugin's custom CSS is only loaded on its own settings page,
	 * preventing unnecessary asset loading on other admin pages.
	 *
	 * @since 1.0.0
	 * @param string $hook The filename of the current admin page, provided by WordPress.
	 */
	public function enqueue_admin_styles( $hook ) {
	      $current_screen = get_current_screen();
	      if ($current_screen && $current_screen->id === 'toplevel_page_paystar-connect-settings') {
	          wp_enqueue_script(
	              'paystar-connect-bulk-sync',
	              PAYSTAR_CONNECT_URL . 'assets/js/admin-bulk-sync.js',
	              ['jquery'],
	              PAYSTAR_CONNECT_VERSION,
	              true
	          );
	      }
	       global $pagenow, $post_type;
	       if (class_exists('WooCommerce') && is_admin() && $pagenow === 'post.php' && $post_type === 'product') {
	           wp_enqueue_script(
	               'paystar-connect-wc-sync',
	               PAYSTAR_CONNECT_URL . 'integrations/woocommerce/assets/js/admin-wc-sync.js',
	               ['jquery'],
	               PAYSTAR_CONNECT_VERSION,
	               true
	           );
	       }
	 // This conditional check ensures that our custom styles are only loaded on the
	 // plugin's settings page, improving performance and avoiding conflicts.
	 if ( 'toplevel_page_paystar-connect-settings' !== $hook ) {
	  return;
	 }
		
		// Enqueue the main admin stylesheet for the settings page.
		wp_enqueue_style(
			'paystar-connect-admin-style',
			PAYSTAR_CONNECT_URL . 'assets/css/admin.css',
			[],
			PAYSTAR_CONNECT_VERSION
		);

		// Enqueue custom fonts for a consistent look and feel in the admin area.
		wp_enqueue_style(
			'paystar-connect-admin-fonts',
			PAYSTAR_CONNECT_URL . 'assets/css/admin-fonts.css',
			[],
			PAYSTAR_CONNECT_VERSION
		);

		// Enqueue the admin script for the settings page.
		wp_enqueue_script(
			'paystar-connect-admin-settings',
			PAYSTAR_CONNECT_URL . 'assets/js/admin-settings.js',
			[ 'jquery' ],
			PAYSTAR_CONNECT_VERSION,
			true
		);
		// Add inline CSS to change the menu icon when the menu item is active or hovered.
		$full_color_svg = '<svg version="1.2" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="512" height="512"><style>.a{fill:#00c1a4}.b{fill:#fe0202}</style><path class="a" d="m101.6 422.1c0.6-6.2 1.2-12.6 1.7-18.8 0.9-9.8 1.9-19.7 2.9-29.6l3.4-36.2 2.6-29c1.2-11.9 2.4-24 3.6-36.2 0.3-4.2 0.7-8.4 1.2-12.4 0-1.9-0.9-2.4-2.1-3l-45.9-25.3-63.3-34.3c-3.5-2-5.7-5.9-5.7-9.9 0-0.4 0-0.9 0-1.2 0.5-5.2 4.2-9.4 9.4-10.6 7.8-1.2 14.8-2 22.2-3.1l39.4-5.2c12-1.7 23.9-3.3 36-4.8 14.5-2.1 29.1-3.9 43.8-5.8 1.8 0 2.3-1.5 3-2.7l33.3-57.3 39.8-68.5c1.8-3.4 5-5.9 8.7-6.9 4.4-1.6 13.1 3.8 13.1 9.2 0 0.7 0 1.4 0 2.2 0 14.1 0.9 28.3 2.6 42.2 2 19.4 6.5 38.6 13.6 56.6 5.5 14 18.8 23.4 34 23.7 4.2 0.5 8.4 0.7 12.6 0.7 4.2 0 8.4-0.2 12.6-0.7 0 1.5-1.7 1.7-2.5 2-39.9 24.8-76.6 54.7-108.9 88.9-18 20.2-34.5 41.9-49.1 64.8-21.2 32.5-40.1 66.8-56.2 102.1-1.4 3.6-3.4 6.8-5.9 9.5z"/><path class="b" d="m376 237.2c-0.4 12.4-1.9 24.7-4.2 37q-0.6 2.8-0.6 5.6 0 2.7 0.6 5.5c0.6 4.9 4.8 8.4 9.7 8.4q0.3 0 0.5 0c5.2 0 10.3-1.7 14.5-4.7 7.2-5.2 14-11.4 19.7-18.2 19.8-22 36-47.1 48.1-74.2 0.9-1.7 1.7-3.2 2.5-4.7 3.1-7.4 9.8-12.5 17.7-13.3 8.1-1.5 16.2-2.5 24.2-2.7 3.9 0 3.9 0 2.2 3.4-17 34.3-37.8 66.6-62.2 96.1-12.8 16-24.6 32.6-38.7 47.3-12.7 13.8-27.6 25.4-44.1 34.3-4.7 2.5-10 4-15.4 4.9-0.8 0.1-1.8 0.1-2.6 0.1-7.8 0-14.7-5.2-16.7-12.6-2-8.6-3-17.5-3-26.4-0.4-8.6-1.4-17.3-3.1-25.7-0.5-2.2-1.1-4.3-2-6.3-0.8-3.7-4.2-6.2-7.9-6.2q-1.5 0-3 0.5c-5.7 1.5-11.1 3.7-16.2 6.7-15.8 9.1-30.6 19.9-44.1 32.2-31.4 27.7-62.2 56.2-90.7 86.8-12.8 13.6-24.9 28-36.5 42.8-2.2 3-4.4 6-6.2 9.4-0.9 1.3-1.9 2.5-2.7 3.9-1 1-2.2 2.1-3.7 1.5-1.2-0.7-2-1.9-2-3.2 0-0.2 0-0.5 0-0.7 0.5-7.2 2.3-14.5 5.2-21 13.1-26.3 29.4-51 48.3-73.6 18.7-24 39.7-45.9 59.2-69 25.3-28.6 53-55.2 82.5-79.4 9.9-9.3 20.7-17.8 32.1-25.3 3.9-2.5 8.1-4.3 12.5-5.7 1.5-0.5 3.2-0.8 4.9-0.8 5.5 0 10.6 2.8 13.3 7.7 2.8 4.7 4.8 10 5.8 15.3 1.6 7.1 2.2 14.5 2.2 21.7q0 1.6 0 3.1z"/><path class="b" d="m280.9 491.5c0.1-4.4 0.8-9 1.8-13.3q0.5-2.3 0.5-4.6c0-7.4-3.7-14.3-9.7-18.5-3.2-2.5-6.8-4.5-10.1-6.9-2.2-1.5-4.4-3.4-6.1-5.4-2.5-2.5-3.9-6-3.9-9.6 0-2.8 0.9-5.7 2.6-8 5-8.1 13.9-13.2 23.5-13.3 6.1-0.5 12.1-0.7 18.2-0.7 7.6 0 15.3 0.3 22.9 1 5.2 0.5 10.4 0.8 15.8 0.8q11.1 0 22.2-1.5c3.1-0.7 6.1-1.5 8.9-2.7 0.7-0.3 1.2-0.5 1.9-0.5 1.2 0 2.4 0.5 3.2 1.5 0.5 0.9 0.8 1.7 0.8 2.7 0 1-0.3 1.9-0.8 2.7-2.4 3.2-5.4 5.7-8.8 7.6-6.9 4.5-14.8 7.1-23 7.7-12 1.2-24.1 3.2-35.9 6.1-3.5 0.8-6.7 2.2-9.7 4.2-1.5 0.8-2.5 2.4-2.5 4 0 0.7 0.1 1.6 0.5 2.1 1.1 2.8 1.8 5.9 1.8 8.9 0 2.2-0.3 4.4-1 6.4-1.5 7-4.5 13.9-8.6 20l-5 8.4 0.1 0.4z"/></svg>';
		$full_color_svg_data_uri = 'data:image/svg+xml;base64,' . base64_encode( $full_color_svg );
		$dynamic_css = "
				#adminmenu #toplevel_page_paystar-connect-settings.current .wp-menu-image,
				#adminmenu #toplevel_page_paystar-connect-settings.wp-has-current-submenu .wp-menu-image,
				#adminmenu #toplevel_page_paystar-connect-settings:hover .wp-menu-image {
					background-image: url('{$full_color_svg_data_uri}') !important;
					background-position: center !important;
					background-repeat: no-repeat !important;
					background-size: 25px !important;
				}
		";
		wp_add_inline_style( 'paystar-connect-admin-style', $dynamic_css );

	}

	/**
	 * Adds the settings page to the WordPress admin menu.
	 *
	 * @since 1.1.0
	 */
	public function add_plugin_page() {
		add_menu_page(
			'تنظیمات پی‌استار', // Title of the page.
			'پی‌استار کانکت', // Menu title.
			'manage_options', // Capability required to access this menu.
			'paystar-connect-settings', // Slug for the menu page.
			[ new PayStar_Connect_Settings(), 'create_admin_page' ], // Callback function to render the settings page.
			$this->get_menu_icon(), // Icon for the menu item.
			58.6 // Position in the menu order.
		);
	}

	/**
	 * Returns the base64-encoded SVG for the custom admin menu icon.
	 *
	 * @since 1.1.0
	 * @return string The SVG data URI for the monochrome icon.
	 */
	private function get_menu_icon() {
		$svg_monochrome = '<svg version="1.2" xmlns="http://www.w3.org/2000/svg" fill="#a7aaad" viewBox="0 0 512 512" width="512" height="512"><path d="m101.6 422.1c0.6-6.2 1.2-12.6 1.7-18.8 0.9-9.8 1.9-19.7 2.9-29.6l3.4-36.2 2.6-29c1.2-11.9 2.4-24 3.6-36.2 0.3-4.2 0.7-8.4 1.2-12.4 0-1.9-0.9-2.4-2.1-3l-45.9-25.3-63.3-34.3c-3.5-2-5.7-5.9-5.7-9.9 0-0.4 0-0.9 0-1.2 0.5-5.2 4.2-9.4 9.4-10.6 7.8-1.2 14.8-2 22.2-3.1l39.4-5.2c12-1.7 23.9-3.3 36-4.8 14.5-2.1 29.1-3.9 43.8-5.8 1.8 0 2.3-1.5 3-2.7l33.3-57.3 39.8-68.5c1.8-3.4 5-5.9 8.7-6.9 4.4-1.6 13.1 3.8 13.1 9.2 0 0.7 0 1.4 0 2.2 0 14.1 0.9 28.3 2.6 42.2 2 19.4 6.5 38.6 13.6 56.6 5.5 14 18.8 23.4 34 23.7 4.2 0.5 8.4 0.7 12.6 0.7 4.2 0 8.4-0.2 12.6-0.7 0 1.5-1.7 1.7-2.5 2-39.9 24.8-76.6 54.7-108.9 88.9-18 20.2-34.5 41.9-49.1 64.8-21.2 32.5-40.1 66.8-56.2 102.1-1.4 3.6-3.4 6.8-5.9 9.5z"/><path d="m376 237.2c-0.4 12.4-1.9 24.7-4.2 37q-0.6 2.8-0.6 5.6 0 2.7 0.6 5.5c0.6 4.9 4.8 8.4 9.7 8.4q0.3 0 0.5 0c5.2 0 10.3-1.7 14.5-4.7 7.2-5.2 14-11.4 19.7-18.2 19.8-22 36-47.1 48.1-74.2 0.9-1.7 1.7-3.2 2.5-4.7 3.1-7.4 9.8-12.5 17.7-13.3 8.1-1.5 16.2-2.5 24.2-2.7 3.9 0 3.9 0 2.2 3.4-17 34.3-37.8 66.6-62.2 96.1-12.8 16-24.6 32.6-38.7 47.3-12.7 13.8-27.6 25.4-44.1 34.3-4.7 2.5-10 4-15.4 4.9-0.8 0.1-1.8 0.1-2.6 0.1-7.8 0-14.7-5.2-16.7-12.6-2-8.6-3-17.5-3-26.4-0.4-8.6-1.4-17.3-3.1-25.7-0.5-2.2-1.1-4.3-2-6.3-0.8-3.7-4.2-6.2-7.9-6.2q-1.5 0-3 0.5c-5.7 1.5-11.1 3.7-16.2 6.7-15.8 9.1-30.6 19.9-44.1 32.2-31.4 27.7-62.2 56.2-90.7 86.8-12.8 13.6-24.9 28-36.5 42.8-2.2 3-4.4 6-6.2 9.4-0.9 1.3-1.9 2.5-2.7 3.9-1 1-2.2 2.1-3.7 1.5-1.2-0.7-2-1.9-2-3.2 0-0.2 0-0.5 0-0.7 0.5-7.2 2.3-14.5 5.2-21 13.1-26.3 29.4-51 48.3-73.6 18.7-24 39.7-45.9 59.2-69 25.3-28.6 53-55.2 82.5-79.4 9.9-9.3 20.7-17.8 32.1-25.3 3.9-2.5 8.1-4.3 12.5-5.7 1.5-0.5 3.2-0.8 4.9-0.8 5.5 0 10.6 2.8 13.3 7.7 2.8 4.7 4.8 10 5.8 15.3 1.6 7.1 2.2 14.5 2.2 21.7q0 1.6 0 3.1z"/><path d="m280.9 491.5c0.1-4.4 0.8-9 1.8-13.3q0.5-2.3 0.5-4.6c0-7.4-3.7-14.3-9.7-18.5-3.2-2.5-6.8-4.5-10.1-6.9-2.2-1.5-4.4-3.4-6.1-5.4-2.5-2.5-3.9-6-3.9-9.6 0-2.8 0.9-5.7 2.6-8 5-8.1 13.9-13.2 23.5-13.3 6.1-0.5 12.1-0.7 18.2-0.7 7.6 0 15.3 0.3 22.9 1 5.2 0.5 10.4 0.8 15.8 0.8q11.1 0 22.2-1.5c3.1-0.7 6.1-1.5 8.9-2.7 0.7-0.3 1.2-0.5 1.9-0.5 1.2 0 2.4 0.5 3.2 1.5 0.5 0.9 0.8 1.7 0.8 2.7 0 1-0.3 1.9-0.8 2.7-2.4 3.2-5.4 5.7-8.8 7.6-6.9 4.5-14.8 7.1-23 7.7-12 1.2-24.1 3.2-35.9 6.1-3.5 0.8-6.7 2.2-9.7 4.2-1.5 0.8-2.5 2.4-2.5 4 0 0.7 0.1 1.6 0.5 2.1 1.1 2.8 1.8 5.9 1.8 8.9 0 2.2-0.3 4.4-1 6.4-1.5 7-4.5 13.9-8.6 20l-5 8.4 0.1 0.4z"/></svg>';
		return 'data:image/svg+xml;base64,' . base64_encode( $svg_monochrome );
	}

	/**
	 * Run the loader.
	 *
	 * This method is the primary entry point that gets the plugin running.
	 * It's called from the main plugin file after the class is instantiated.
	 *
	 * @since 1.0.0
	 */
	public function run() {
		// The plugin is now initialized and running.
		// This method can be used to trigger actions that need to happen after setup.
	}

	/**
	 * Displays admin notices fetched from the Dinor Digital remote server.
	 * Checks plugin version to show relevant update notifications.
	 *
	 * @since 2.3.0
	 */
	public function dinor_display_remote_admin_notice() {
		// Only show notices on the Dashboard and Plugins pages.
		$screen = get_current_screen();
		if ( ! $screen || ! in_array( $screen->base, ['dashboard', 'plugins'] ) ) {
			return;
		}

		$transient_key = 'dinor_plugins_remote_notice';
		$notice_data = get_transient( $transient_key );

		if ( false === $notice_data ) {
			$api_url = 'https://dinor-plugins-api.dinordigitalstudio.workers.dev/status';
			$request_url = add_query_arg( 'plugin', 'paystar-connect', $api_url );
			
			$response = wp_remote_get( $request_url, ['timeout' => 10] );

			if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
				// If the request fails, set a short-lived transient to prevent repeated failed requests.
				set_transient( $transient_key, 'error', HOUR_IN_SECONDS );
				return;
			}

			$notice_data = json_decode( wp_remote_retrieve_body( $response ), true );
			// Cache the response for 12 hours.
			set_transient( $transient_key, $notice_data, 24 * HOUR_IN_SECONDS );
		}

		// Proceed only if there is a message structure.
		if ( ! empty( $notice_data['message'] ) && is_array( $notice_data['message'] ) ) {
			$message = $notice_data['message'];

			// Check if a 'latest_version' is specified in the message.
			if ( isset( $message['latest_version'] ) ) {
				
				// Compare the plugin's current version with the latest version from the message.
				// The notice will be shown if the current version is LESS THAN the latest version.
				if ( defined('PAYSTAR_CONNECT_VERSION') && version_compare( PAYSTAR_CONNECT_VERSION, $message['latest_version'], '<' ) ) {
					
					$type = ! empty( $message['type'] ) ? esc_attr( $message['type'] ) : 'warning';
					$text = wp_kses_post( $message['text'] );

					echo '<div class="notice notice-' . esc_attr( $type ) . ' is-dismissible"><p>' . wp_kses_post( $text ) . '</p></div>';
				}
			}
		}
	}
}