<?php
/**
 * Handles the payment processing and callback logic for the Paystar Gateway for Gravity Forms.
 *
 * This class manages the entire payment lifecycle: initiating the payment after form submission,
 * handling the callback from the payment gateway, verifying the transaction, and displaying
 * a final confirmation or error message to the user.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/integrations/gravity-forms
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Guardian Clause: Stop loading if Gravity Forms is not active to prevent fatal errors.
if ( ! class_exists( 'GFCommon' ) ) {
    return;
}

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
} 

/**
 * Manages the payment lifecycle for Gravity Forms and Paystar.
 */
class PayStar_GF_Gateway_Handler {

	private $logger;

	/**
	 * Constructor.
	 *
	 * Sets up the necessary hooks for the three main phases of the payment process.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		if ( ! class_exists( 'PayStar_Connect_Logger' ) ) {
			require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-logger.php';
		}
		$this->logger = new PayStar_Connect_Logger();

		// Phase 1: Use gform_after_submission to start the payment process.
		add_action( 'gform_after_submission', [ $this, 'process_payment' ], 10, 2 );

		// Phase 2: Use init to create a centralized callback handler that listens for returns from the gateway.
		add_action( 'init', [ $this, 'handle_callback' ] );

		// Phase 3: Display the final payment message (success or failure).
		// This hooks into gform_pre_render to modify the form object just before it is displayed,
		// allowing us to replace the form with a confirmation or add an error message.
		add_filter( 'gform_pre_render', [ $this, 'maybe_display_payment_message' ] );
	}

	/**
	 * Phase 1: Initiates the payment process after a form is submitted.
	 *
	 * This function is triggered after a form with the Paystar gateway enabled is submitted.
	 * It gathers the necessary data, calls the Paystar API to create a transaction,
	 * and redirects the user to the payment gateway.
	 *
	 * @since 1.0.0
	 *
	 * @param array $entry The entry object created by Gravity Forms.
	 * @param array $form  The form object that was submitted.
	 * @return void
	 */
	public function process_payment( $entry, $form ) {

		// Check if the Paystar gateway is enabled in the form's settings.
		if ( empty( $form['paystar_gf_enable'] ) ) {
			return;
		}

		$this->logger->log( '--------------------------------' );
		$this->logger->log( 'Gravity Forms: Initiating payment process for Entry ID: ' . $entry['id'] );

		// Retrieve the main plugin settings to get API credentials.
		$main_options = get_option( 'paystar_connect_settings' );
		if ( empty( $main_options['gateway_id'] ) || empty( $main_options['sign_key'] ) ) {
			// If credentials are missing, add an error note to the entry for the admin to see.
			$error_message = 'خطا: درگاه پی‌استار کانکت فعال است، اما شناسه درگاه یا کلید امضا وجود ندارد.';
			GFAPI::add_note( $entry['id'], 'درگاه پی‌استار کانکت', $error_message, 'error' );
			$this->logger->log( 'Gravity Forms: Payment failed for Entry ID: ' . $entry['id'] . '. Reason: ' . $error_message );
			return;
		}

		// Get the total amount from the form entry. This can come from a product field or a total field.
		$amount = GFCommon::get_order_total( $form, $entry );

		// The Paystar API expects the amount in Rials. If the currency is in Tomans (IRT), convert it.
		if ( GFCommon::get_currency() === 'IRT' ) {
			$amount *= 10;
		}

		// Update the entry to show that payment processing has started.
		GFAPI::update_entry_property( $entry['id'], 'payment_status', 'Processing' );

		// Construct the callback URL where Paystar will redirect the user after payment.
		// We add unique query parameters to identify our callback requests.
		$callback_base = add_query_arg( [ 'paystar_gf_callback' => '1', 'entry_id' => $entry['id'] ], $entry['source_url'] );
		$callback_url = html_entity_decode( $callback_base );		

		// Instantiate the API handler and create the payment request.
		$api      = new PayStar_Connect_API();
		$extra_data = [];
		if ( ! empty( $main_options['base_domain'] ) && 'starshop' === $main_options['base_domain'] ) {
			$first_name_field_id = rgar( $form, 'paystar_gf_fname_field' );
			$last_name_field_id  = rgar( $form, 'paystar_gf_lname_field' );

			$extra_data['first_name'] = rgar( $entry, $first_name_field_id );
			$extra_data['last_name']  = rgar( $entry, $last_name_field_id );
			$extra_data['products']   = [
				[
					'code'     => 'GF_' . $form['id'],
					'price'    => (string) $amount,
					'quantity' => 1,
				],
			];
		}
		$response = $api->create( $amount, $entry['id'], $callback_url, $extra_data );

		if ( $response['status'] ) {
			// If the transaction was created successfully, redirect the user to the payment page.
			GFAPI::add_note( $entry['id'], 'درگاه پی‌استار کانکت', 'کاربر به درگاه پرداخت هدایت شده است. توکن: ' . $response['token'], 'note' );
			$payment_url = $response['payment_url'] . '?token=' . $response['token'];
			wp_redirect( $payment_url );
			exit; // Always exit after a redirect to prevent further script execution.
		} else {
			// If the transaction creation failed, update the entry and show an error.
			GFAPI::update_entry_property( $entry['id'], 'payment_status', 'Failed' );
			$error_message = 'خطا هنگام ایجاد پرداخت: ' . $response['message'];
			GFAPI::add_note( $entry['id'], 'درگاه پی‌استار کانکت', $error_message, 'error' );
			$this->logger->log( 'Gravity Forms: Payment creation failed for Entry ID: ' . $entry['id'] . '. Reason: ' . $response['message'] );
			$this->finalize_and_redirect( $entry, $response['message'] );
		}
	}

	/**
	 * Phase 2: Handles the callback from the Paystar gateway.
	 *
	 * This function listens for requests to our specific callback URL. It processes the
	 * data sent back by Paystar, verifies the payment, and updates the entry status.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function handle_callback() {
		// Check if this is a callback request from our Gravity Forms integration.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( empty( $_GET['paystar_gf_callback'] ) || empty( $_GET['entry_id'] ) ) {
			return;
		}
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$this->logger->log( 'Gravity Forms Callback: Received data for Entry ID ' . absint( wp_unslash( $_GET['entry_id'] ) ) . ': ' . wp_json_encode( $_GET, JSON_UNESCAPED_UNICODE ) );
		
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$entry_id = absint( wp_unslash( $_GET['entry_id'] ) );
		$entry    = GFAPI::get_entry( $entry_id );

		// Ensure the entry exists.
		if ( is_wp_error( $entry ) ) {
			wp_die( 'خطا: اطلاعات ورودی یافت نشد.' ); // "Error: Entry information not found."
		}

		$form = GFAPI::get_form( $entry['form_id'] );

		// Handle cases where the user cancelled the payment or it failed initially.
		// Paystar sends a status of '1' for success.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['status'] ) && sanitize_text_field( wp_unslash( $_GET['status'] ) ) != 1 ) {
			GFAPI::update_entry_property( $entry_id, 'payment_status', 'Cancelled' );
			$note = 'پرداخت توسط کاربر لغو شد.'; // "Payment cancelled by user."
			GFAPI::add_note( $entry_id, 'درگاه پی‌استار کانکت', $note, 'note' );
			$this->logger->log( 'Gravity Forms: Payment cancelled by user for Entry ID: ' . $entry_id );
			$message = 'شما پرداخت را لغو کرده‌اید. در صورت تمایل برای تکمیل خرید، لطفاً دوباره تلاش کنید.'; // "You have cancelled the payment..."
			$this->finalize_and_redirect( $entry, $message );
		}

		// If we've reached here, the payment appears to be successful. Now, we must verify it.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$ref_num       = isset( $_GET['ref_num'] ) ? sanitize_text_field( wp_unslash( $_GET['ref_num'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$card_number   = isset( $_GET['card_number'] ) ? sanitize_text_field( wp_unslash( $_GET['card_number'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$tracking_code = isset( $_GET['tracking_code'] ) ? sanitize_text_field( wp_unslash( $_GET['tracking_code'] ) ) : '';
		$amount        = GFCommon::get_order_total( $form, $entry );

		// Ensure the amount for verification is also in Rials.
		if ( GFCommon::get_currency() === 'IRT' ) {
			$amount *= 10;
		}

		// Call the API to verify the transaction.
		$api                 = new PayStar_Connect_API();
		$verification_result = $api->verify( $amount, $ref_num, $card_number, $tracking_code );

		if ( isset( $verification_result['status'] ) && $verification_result['status'] == 1 ) {
			// --- Verification Successful ---
			GFAPI::update_entry_property( $entry_id, 'payment_status', 'Paid' );
			GFAPI::update_entry_property( $entry_id, 'transaction_id', $ref_num );
			GFAPI::update_entry_property( $entry_id, 'payment_date', gmdate( 'Y-m-d H:i:s' ) );
			$main_options  = get_option( 'paystar_connect_settings' );
			$provider_name = 'پی‌استار'; // Default value
			if ( ! empty( $main_options['base_domain'] ) ) {
				if ( $main_options['base_domain'] === 'starshop' ) {
					$provider_name = 'استارشاپ (اتصال به فروشگاه‌ساز)';
				} elseif ( $main_options['base_domain'] === 'starshop_direct' ) {
					$provider_name = 'استارشاپ';
				}
			}
			$note = sprintf(
				"تایید پرداخت موفق: | درگاه: %s | کد رهگیری درگاه: %s | کد رهگیری بانکی: %s | شماره کارت: %s",
				$provider_name,
				$ref_num,
				$tracking_code,
				$card_number
			);
			GFAPI::add_note( $entry_id, 'درگاه پی‌استار کانکت', $note, 'success' );

			// Store the dynamic success message and redirect.
			$success_message = 'پرداخت شما با موفقیت انجام شد. کد رهگیری: ' . $tracking_code; // "Your payment was successful..."

			// Announce the successful payment to other plugins that might be listening.
			do_action( 'gform_post_payment_completed', $entry, [] );

			// Redirect back to the original form page.
			$this->finalize_and_redirect( $entry, $success_message );

		} else {
			// --- Verification Failed ---
			GFAPI::update_entry_property( $entry_id, 'payment_status', 'Failed' );
			$error_message = $this->get_verification_error_message( $verification_result );
			$note          = 'تایید پرداخت ناموفق بود. دلیل: ' . $error_message; // "Payment verification failed. Reason: ..."
			GFAPI::add_note( $entry_id, 'درگاه پی‌استار کانکت', $note, 'error' );
			$this->logger->log( 'Gravity Forms: Payment verification failed for Entry ID: ' . $entry_id . '. Reason: ' . $error_message );
			$this->finalize_and_redirect( $entry, $error_message );
		}
	}

	/**
	 * Phase 3: Displays the final payment message by modifying the form object before it renders.
	 *
	 * On failure, it adds an error message above the form.
	 * On success, it replaces the entire form with the confirmation message.
	 *
	 * @since 1.0.0
	 *
	 * @param array $form The form object passed by the 'gform_pre_render' filter.
	 * @return array The modified form object.
	 */
	public function maybe_display_payment_message( $form ) {
		// Check for the entry ID we added to the query string on redirect.
		if ( empty( $form['id'] ) || ! isset( $_GET['gf_entry_id'] ) || ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_key( $_GET['_wpnonce'] ), 'paystar_gf_result_' . absint( $_GET['gf_entry_id'] ) ) ) {
			return $form;
		}

		$entry_id = absint( wp_unslash( $_GET['gf_entry_id'] ) );
		if ( ! $entry_id ) {
			return $form;
		}

		// Retrieve the message we stored in the transient during the callback.
		$message = get_transient( 'paystar_gf_message_' . $entry_id );

		// If no message exists (e.g., transient expired or was never set), do nothing.
		if ( ! $message ) {
			return $form;
		}

		// We have a message. Now, get the entry to check its final status.
		$entry = GFAPI::get_entry( $entry_id );
		if ( is_wp_error( $entry ) || rgar( $entry, 'form_id' ) != $form['id'] ) {
			return $form;
		}

		$payment_status = rgar( $entry, 'payment_status' );

		if ( $payment_status === 'Paid' ) {
			// --- SUCCESS SCENARIO ---

			// Get the official confirmation content from Gravity Forms (e.g., "Thanks for contacting us!").
			$confirmation_content = GFFormDisplay::handle_confirmation( $form, $entry, false );

			// Create our custom success message div.
			$success_div = '<div class="gform_payment_message gform_payment_message--success">' . esc_html( $message ) . '</div>';

			// Create a "Return to Form" button.
			$form_page_url = esc_url( $entry['source_url'] );
			$button_html = '<br><div class="gform_payment_message__button-wrap"><a href="' . $form_page_url . '" class="gform_payment_message__button">بازگشت به فرم پرداخت</a></div>'; // "Return to Payment Form"

			// Combine our message with the default confirmation and the button.
			$final_confirmation = $success_div . $confirmation_content . $button_html;

			// To prevent the user from resubmitting, we replace the entire form with our confirmation message.
			$form['fields'] = []; // Empty all fields.
			$form['button'] = []; // Remove the submit button.
			$form['pagination'] = []; // Remove pagination.

			// Create a new HTML field to hold our final confirmation content.
			$confirmation_field          = new GF_Field_HTML();
			$confirmation_field->type    = 'html';
			$confirmation_field->content = $final_confirmation;
			$confirmation_field->formId  = $form['id'];
			$form['fields'][] = $confirmation_field; // Add only our confirmation field to the form.

		} elseif ( in_array( $payment_status, [ 'Failed', 'Cancelled' ] ) ) {
			// --- FAILURE SCENARIO ---
			// Create an HTML field to display the error message above the form.
			$error_field          = new GF_Field_HTML();
			$error_field->type    = 'html';
			$error_field->content = '<div class="gform_payment_message gform_payment_message--error">' . esc_html( $message ) . '</div>';
			$error_field->formId  = $form['id'];

			// Add the error message field to the very top of the form so the user sees it first.
			array_unshift( $form['fields'], $error_field );
		}

		// The message has been displayed, so delete the transient to prevent it from showing again.
		delete_transient( 'paystar_gf_message_' . $entry_id );

		return $form;
	}

	/**
	 * Translates verification API status codes into user-friendly Persian messages.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param array $response The API response from the verify call.
	 * @return string The user-friendly error message.
	 */
	private function get_verification_error_message( $response ) {
		$status = $response['status'] ?? -99;
		switch ( $status ) {
			case -2:
				return 'درگاه پرداخت در حال حاضر غیرفعال است. لطفاً با پشتیبانی تماس بگیرید.'; // "Gateway is currently disabled..."
			case -4:
				return 'مبلغ تراکنش از سقف مجاز درگاه بیشتر است.'; // "Transaction amount exceeds the gateway limit."
			case -5:
				return 'اطلاعات این تراکنش نامعتبر است. لطفاً دوباره تلاش کنید.'; // "Transaction information is invalid..."
			case -6:
				return 'این تراکنش قبلاً تأیید شده و امکان بررسی مجدد آن وجود ندارد.'; // "This transaction has already been verified..."
			case -98:
				return 'پرداخت در شبکه بانکی ناموفق بود. لطفاً دوباره تلاش کنید.'; // "Payment failed in the banking network..."
			case -99:
			default:
				return $response['message'] ?? 'خطای کلی در سیستم رخ داده است. لطفاً با پشتیبانی تماس بگیگیرید.'; // "A general system error occurred..."
		}
	}

	/**
	 * Finalizes the payment process by storing a result message and redirecting the user.
	 *
	 * This helper function centralizes the logic for both successful and failed payments
	 * to avoid code duplication, adhering to the DRY principle.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param array  $entry   The entry object.
	 * @param string $message The message to display to the user.
	 * @return void
	 */
	private function finalize_and_redirect( $entry, $message ) {
		// Step 1: Store the message in a transient to pass it to the confirmation page.
		set_transient( 'paystar_gf_message_' . $entry['id'], $message, 60 * 5 );

		// Step 2: Generate the secure redirect URL with entry ID and a nonce.
		$redirect_url = add_query_arg( [ 'gf_entry_id' => $entry['id'] ], $entry['source_url'] );
		$redirect_url = html_entity_decode( wp_nonce_url( $redirect_url, 'paystar_gf_result_' . $entry['id'], '_wpnonce' ) );

		// Step 3: Redirect the user and stop script execution.
		wp_redirect( $redirect_url );
		exit;
	}
}
