<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Handles the synchronization of Gravity Forms with Starshop products.
 *
 * @since 1.0.0
 */
class GF_Starshop_Sync_Handler implements Paystar_Connect_Sync_Handler_Interface {

	public const BATCH_SYNC_ACTION = 'paystar_connect_sync_gf_form';
	private      $api_service;
	private      $sync_manager;

	public function __construct() {
		$options = get_option( 'paystar_connect_settings' );

		if (
			class_exists( 'GFAPI' ) &&
			! empty( $options['base_domain'] ) && 'starshop' === $options['base_domain'] &&
			! empty( $options['store_id'] ) && ! empty( $options['store_api_key'] )
		) {
			$this->api_service  = new PayStar_Connect_Starshop_Product_API();
			$this->sync_manager = Paystar_Connect_Sync_Manager::get_instance();
			add_action( 'gform_after_save_form', [ $this, 'handle_form_save' ], 10, 2 );
			add_action( 'gform_post_save_feed', [ $this, 'handle_feed_save' ], 10, 2 );
			add_action( self::BATCH_SYNC_ACTION, [ $this, 'process_sync_action' ], 10, 1 );
		}
	}

	public function get_id() {
		return 'gf-forms';
	}

	public function get_name() {
		return __( 'گرویتی‌فرم', 'paystar-connect' );
	}

	public function get_all_item_ids() {
		if ( ! class_exists( 'GFAPI' ) ) {
			return [];
		}
		$forms                           = GFAPI::get_forms( true, false, 'title' );
		$form_ids_with_paystar_enabled = [];

		foreach ( $forms as $form ) {
			if ( rgar( $form, 'paystar_gf_enable' ) ) {
				$form_ids_with_paystar_enabled[] = $form['id'];
			}
		}
		return $form_ids_with_paystar_enabled;
	}

	public function handle_form_save( $form, $is_new ) {
		$this->sync_form_to_starshop( $form['id'] );
	}

	public function handle_feed_save( $feed, $form_id ) {
		if ( rgar( $feed, 'addon_slug' ) === 'paystar-connect' ) {
			$this->sync_form_to_starshop( $form_id );
		}
	}


	public function process_sync_action( $args ) {
		$item_id = is_array( $args ) && isset( $args['item_id'] ) ? $args['item_id'] : $args;
		$run_id  = is_array( $args ) && isset( $args['run_id'] ) ? $args['run_id'] : null;
		$this->sync_form_to_starshop( $item_id, $run_id );
	}

	public function sync_form_to_starshop( $form_id, $run_id = null ) {
		$form = GFAPI::get_form( $form_id );
		if ( ! $form ) {
			return;
		}

		$is_paystar_active = rgar( $form, 'paystar_gf_enable' );
		$final_status      = $is_paystar_active && $form['is_active'];

		$api_data = [
			'title'        => $form['title'],
			'product_code' => 'GF_' . $form['id'],
			'status'       => $final_status ? '1' : '0',
			'price_type'   => '2',
			'unlimited'    => '1',
			'quantity'     => '0',
		];

		$starshop_product_id = rgar( $form, '_starshop_product_id' );
		$result              = null;

		if ( ! $run_id ) {
			$run_id = get_option( 'starshop_sync_current_run_id' );
		}

		if ( ! empty( $starshop_product_id ) ) {
			$result = $this->api_service->update_product( $starshop_product_id, $api_data );
			if ( ! is_wp_error( $result ) ) {
				$this->sync_manager->increment_sync_stat( $run_id, $form_id, 'updated' );
			}
		} elseif ( $final_status ) {
			$result = $this->api_service->create_product( $api_data );
			if ( ! is_wp_error( $result ) && ! empty( $result['data']['id'] ) ) {
				$form['_starshop_product_id'] = $result['data']['id'];
				GFAPI::update_form( $form );
				$this->sync_manager->increment_sync_stat( $run_id, $form_id, 'created' );
			}
		}

		if ( is_wp_error( $result ) ) {
			$clear_id_callback = function () use ( &$form ) {
				$form['_starshop_product_id'] = '';
				GFAPI::update_form( $form );
			};

			$update_id_callback = function ( $new_id ) use ( &$form ) {
				$form['_starshop_product_id'] = $new_id;
				GFAPI::update_form( $form );
			};

			$error_details = $this->sync_manager->handle_sync_error(
				$result,
				$api_data,
				$form_id,
				$run_id,
				$clear_id_callback,
				$update_id_callback,
				$this->api_service
			);

			if ( $error_details ) {
				$log_message = sprintf(
					'خطای همگام‌سازی با پلتفرم استارشاپ از %d: %s - %s (راه حل پیشنهادی: %s)',
					$form_id,
					$error_details['title'],
					$error_details['reason'],
					$error_details['solution']
				);
				if ( $run_id ) {
					$this->sync_manager->add_error_log( $run_id, $form_id, $error_details );
					$this->sync_manager->increment_sync_stat( $run_id, $form_id, 'errors' );
				}
			}
		}
	}

	public function process_batch( $batch ) {
		$processed_count = 0;
		foreach ( $batch as $item_id ) {
			$this->process_sync_action( $item_id );
			$processed_count++;
		}
		return [
			'success'         => true,
			'processed_count' => $processed_count,
		];
	}
}