<?php
/**
 * Main WooCommerce Integration Class for Paystar Connect.
 *
 * This class acts as the primary controller for the WooCommerce integration.
 * It is responsible for registering the payment gateway, handling the callback
 * from the Paystar service, and enqueueing necessary admin styles.
 *
 * @package    Paystar_Connect
 * @subpackage Paystar_Connect/integrations/woocommerce
 * @author     Dinor Digital <info@dinordigital.ir>
 * @since      1.0.0
 */

// Guardian Clause: Stop loading if WooCommerce is not active to prevent fatal errors.
if ( ! class_exists( 'WooCommerce' ) ) {
	return;
}

// Exit if accessed directly for security.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
} 

/**
 * Initializes the Paystar gateway for WooCommerce.
 */
class WooCommerce_Integration {

	private $logger;

	/**
	 * Constructor.
	 *
	 * Sets up the necessary hooks for the WooCommerce integration.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		if ( ! class_exists( 'PayStar_Connect_Logger' ) ) {
			require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-logger.php';
		}
		$this->logger = new PayStar_Connect_Logger();
		// This ensures the gateway class is always available when WooCommerce needs it.
		// Declare compatibility with WooCommerce Checkout Blocks.
		add_action( 'before_woocommerce_init', function() {
			if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', PAYSTAR_CONNECT_PATH . 'paystar-connect.php', true );
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', PAYSTAR_CONNECT_PATH . 'paystar-connect.php', true );
			}
		} );

		// This ensures the gateway class is always available when WooCommerce needs it.
		require_once __DIR__ . '/class-wc-gateway-paystar.php';

		// Register Blocks integration (Checkout Block support).
		add_action( 'woocommerce_blocks_loaded', function() {
			// AbstractPaymentMethodType فقط وقتی موجود است که بستهٔ بلوک‌ها نصب/فعال باشد.
			if ( class_exists( '\\Automattic\\WooCommerce\\Blocks\\Payments\\Integrations\\AbstractPaymentMethodType' ) ) {
				require_once __DIR__ . '/class-wc-gateway-paystar-blocks.php';
				add_action( 'woocommerce_blocks_payment_method_type_registration', function( $payment_method_registry ) {
					$payment_method_registry->register( new WC_PayStar_Blocks() );
				} );
			}
		} );

		// Load the Starshop sync handler and instantiate it.
		require_once plugin_dir_path(__FILE__) . 'class-wc-starshop-sync-handler.php';
		new WC_Starshop_Sync_Handler();

		// Add the gateway to the classic list of available payment methods in WooCommerce.
		add_filter( 'woocommerce_payment_gateways', [ $this, 'add_gateway_class' ] );

		// Add custom styles for the gateway's settings page in the admin area.
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_styles' ] );

		// Register the custom API endpoint for handling the payment callback from Paystar.
		// This is the URL the user is redirected back to after a payment attempt.
		add_action( 'woocommerce_api_paystar_connect_callback', [ $this, 'handle_callback' ] );
	}

	/**
	 * Adds the Paystar gateway class to the list of available WooCommerce payment gateways.
	 *
	 * @since 1.0.0
	 *
	 * @param  array $gateways The array of existing gateway classes.
	 * @return array The modified array of gateway classes.
	 */
	public function add_gateway_class( $gateways ) {
		$gateways[] = 'WC_Gateway_PayStar';
		return $gateways;
	}

	/**
	 * Enqueues stylesheets for the WooCommerce admin area.
	 *
	 * This ensures our custom styles are only loaded on our specific settings page,
	 * preventing unnecessary asset loading elsewhere in the admin panel.
	 *
	 * @since 1.0.0
	 *
	 * @param string $hook The filename of the current admin page.
	 * @return void
	 */
	public function enqueue_admin_styles( $hook ) {
		// This condition ensures that the styles are only loaded on the correct settings page
		// by checking the page hook, the 'tab', and the 'section' query parameters.
		if ( 'woocommerce_page_wc-settings' !== $hook ) {
			return;
		}
		// Nonce verification for security
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_key( $_GET['_wpnonce'] ), 'woocommerce-settings' ) ) {
			// Optionally, you could die or handle the error more gracefully.
			// For a settings page, simply returning might be sufficient.
			// wp_die( 'Security check failed' );
		}

		$current_tab     = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : '';
		$current_section = isset( $_GET['section'] ) ? sanitize_text_field( wp_unslash( $_GET['section'] ) ) : '';

		if ( $current_tab !== 'checkout' || $current_section !== 'paystar_connect' ) {
			return;
		}

		$css_url = PAYSTAR_CONNECT_URL . 'integrations/woocommerce/assets/css/wc-paystar.css';
		$version = PAYSTAR_CONNECT_VERSION;

		// Enqueue the specific styles for the WooCommerce settings section.
		wp_enqueue_style( 'paystar-wc-settings-styles', $css_url, [], $version );

		// Also load the main admin font definitions file for a consistent UI.
		$font_css_url = PAYSTAR_CONNECT_URL . 'assets/css/admin-fonts.css';
		wp_enqueue_style( 'paystar-admin-fonts', $font_css_url, [], $version );
	}


	/**
	 * Handles the callback from Paystar after a payment attempt.
	 *
	 * This method is the endpoint for the `woocommerce_api_paystar_connect_callback` action.
	 * It processes the payment result, verifies the transaction with the Paystar API,
	 * and updates the order status accordingly.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function handle_callback() {
		// phpcs:disable WordPress.Security.NonceVerification.Recommended
		// Nonce verification is intentionally skipped here because this is a server-to-server callback from the payment gateway.
		// The authenticity of the request is verified later using the `verify` API call, which is a more secure method for this context.
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$this->logger->log( 'WooCommerce Callback: Received data: ' . wp_json_encode( $_GET, JSON_UNESCAPED_UNICODE ) );

		// 2. Order Validation.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$order_id = isset( $_GET['wc_order_id'] ) ? absint( wp_unslash( $_GET['wc_order_id'] ) ) : 0;
		$order    = wc_get_order( $order_id );

		if ( ! $order || ! $order->needs_payment() ) {
			$this->logger->log( 'WooCommerce Callback: Invalid order or already paid for Order ID: ' . $order_id );
			wc_add_notice( __( 'سفارش نامعتبر است یا قبلاً پرداخت شده است.', 'paystar-connect' ), 'error' );
			wp_redirect( wc_get_checkout_url() );
			exit;
		}

		// 3. Handle User Cancellation (no POST data from Paystar).
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( empty( $_GET['status'] ) ) {
			$order->add_order_note( __( 'کاربر پرداخت را لغو کرده است.', 'paystar-connect' ) );
			wc_add_notice( __( 'پرداخت توسط شما لغو شد.', 'paystar-connect' ), 'notice' );
			wp_redirect( $order->get_cancel_order_url() );
			exit;
		}

		// 4. Sanitize and process payment data.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$status        = isset( $_GET['status'] ) ? (int) sanitize_text_field( wp_unslash( $_GET['status'] ) ) : 0;
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$ref_num       = isset( $_GET['ref_num'] ) ? sanitize_text_field( wp_unslash( $_GET['ref_num'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$card_number   = isset( $_GET['card_number'] ) ? sanitize_text_field( wp_unslash( $_GET['card_number'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$tracking_code = isset( $_GET['tracking_code'] ) ? sanitize_text_field( wp_unslash( $_GET['tracking_code'] ) ) : '';

		// 5. Handle failed payment on the gateway side.
		if ( $status !== 1 ) {
			$order->update_status( 'failed', __( 'پرداخت ناموفق بود یا توسط شما در درگاه لغو شد.', 'paystar-connect' ) );
			wc_add_notice( __( 'پرداخت ناموفق بود یا توسط شما لغو شد.', 'paystar-connect' ), 'error' );
			wp_redirect( wc_get_checkout_url() );
			exit;
		}

		// 6. Verify the successful transaction with Paystar API.
		$amount = paystar_connect_get_amount_by_currency( $order->get_total(), $order->get_currency() );
		$api    = new PayStar_Connect_API();
		$result = $api->verify( $amount, $ref_num, $card_number, $tracking_code );

		if ( isset( $result['status'] ) && $result['status'] === 1 ) {
			// --- PAYMENT SUCCESS ---
			$order->payment_complete( $ref_num );

			$main_options  = get_option( 'paystar_connect_settings' );
			$provider_name = __( 'پی‌استار', 'paystar-connect' );
			if ( ! empty( $main_options['base_domain'] ) ) {
				if ( $main_options['base_domain'] === 'starshop' ) {
					$provider_name = __( 'استارشاپ (متصل به فروشگاه‌ساز)', 'paystar-connect' );
				} elseif ( $main_options['base_domain'] === 'starshop_direct' ) {
					$provider_name = __( 'استارشاپ', 'paystar-connect' );
				}
			}

			/* translators: 1: Gateway name (e.g., Paystar), 2: Gateway tracking code, 3: Bank tracking code, 4: Card number */
			$note = sprintf(
				__( "تأیید پرداخت موفقیت‌آمیز بود:\nدرگاه: <strong>%1\$s</strong>\nکد رهگیری درگاه: %2\$s\nکد رهگیری بانک: %3\$s\nشماره کارت: %4\$s", 'paystar-connect' ),
				$provider_name,
				$ref_num,
				$tracking_code,
				$card_number
			);
			$order->add_order_note( $note );

			// --- Post-Payment Sync for Starshop ---
			if ( isset( $main_options['base_domain'] ) && 'starshop' === $main_options['base_domain'] ) {
				$this->starshop_post_payment_sync( $order );
			}
			// --- End of Post-Payment Sync ---

			wp_redirect( $order->get_checkout_order_received_url() );
			exit;

		} else {
			// --- VERIFICATION FAILED ---
			$error_message = $result['message'] ?? __( 'خطای ناشناخته‌ای هنگام تأیید پرداخت رخ داد.', 'paystar-connect' );
			/* translators: %s: Error message from the payment gateway */
			$order->update_status( 'failed', sprintf( __( 'تأیید پرداخت ناموفق بود. پیام: %s', 'paystar-connect' ), $error_message ) );
			$this->logger->log( 'WooCommerce Callback: Payment verification failed for Order ID: ' . $order_id . '. Reason: ' . $error_message );
			wc_add_notice( $error_message, 'error' );
			wp_redirect( wc_get_checkout_url() );
			exit;
		}
		// phpcs:enable WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Handles the post-payment synchronization with Starshop.
	 *
	 * @param WC_Order $order The order object.
	 */
	private function starshop_post_payment_sync( $order ) {
		try {
			if ( ! class_exists( 'PayStar_Connect_Starshop_Product_API' ) ) {
				require_once PAYSTAR_CONNECT_PATH . 'includes/class-paystar-connect-starshop-product-api.php';
			}
			$product_api = new PayStar_Connect_Starshop_Product_API();

			foreach ( $order->get_items() as $item ) {
				$product = $item->get_product();
				if ( ! $product ) {
					continue;
				}

				$starshop_product_id = get_post_meta( $product->get_id(), '_starshop_product_id', true );

				if ( ! empty( $starshop_product_id ) ) {
					$update_data = [
						'quantity' => $product->get_stock_quantity(),
					];
					$product_api->update_product( $starshop_product_id, $update_data );
				}
			}
		} catch ( Exception $e ) {
			if ( function_exists( 'wc_get_logger' ) ) {
				wc_get_logger()->error( 'Starshop post-payment sync failed for order ' . $order->get_id() . ': ' . $e->getMessage() );
			}
		}
	}
}